const asyncHandler = require("../../../helpers/asyncHandler");
const { slugMaker, isURL } = require("../../../helpers/dataManipulator");
const { getPaginationNextPrev, getNextOffset } = require("../../../helpers/pagination");
const { createNewPost, createPostCategoryPivot, editPostById, getAllPost, getCategoriesOfPost, deletePostCategoryPivot, getPostById, deletePosts, filterPostData } = require("../../../helpers/post");
const { deleteFileAttachmentMulter } = require("../../../multer/multerMiddleware");
const path = require("path");
const fs = require("fs");
const { processVideo, deleteFolderRecursive, downloadVideo } = require("../../../helpers/videoProcessing");
const { generateRandomId } = require("../../../helpers/uniqueID");
const { getUploadById } = require("../../../helpers/uploadVideo");
const { deleteS3Folder, uploadFileToS3, deleteS3File } = require("../../../helpers/s3");
const { getWebSettings } = require("../../../helpers/settings");

//ADMIN GET ALL POST
exports.adminAllPostsGet = asyncHandler(async (req, res, next) => {

    const limit = req.query.limit || parseInt(process.env.LIMIT), currentPage = parseInt(req.query.page) || 1, search = req.query.search, status = req.query.status;

    const { totalCount, contents } = await getAllPost({
        query: { search, categories: req.query.categories?.split(",") || [], status, limit, offset: getNextOffset(currentPage, limit) }
    });

    const { prevPage, nextPage, totalPages } = getPaginationNextPrev({ page: currentPage, limit, totalCount});


    res.json({ status: true, data: { contents, pagination: { currentPage, prevPage, nextPage, totalPages } }})
});

//CREATE NEW POST
exports.adminPostPost = asyncHandler(async (req, res, next) => {

    try{

        const settings = await getWebSettings();
        
        const filterResult = filterPostData({ title: req.body.c_title, description: req.body.c_description, tags: JSON.parse(req.body.c_tags || []).join(" ") }, JSON.parse(settings.website_banned_words));
        if(!filterResult.status) {
            return req.theMode == "submission" ?  filterResult : res.json(filterResult);
        }
        
        let submissionPost;

        if(req.theMode == "submission"){
            submissionPost = await getUploadById(req.query.id || '0');
            if(!submissionPost) return { status: false, message: "Submission post not found"}
        };

        let uploadedFile = req.compressedFiles.find(cf => cf.fieldname == "c_file");
        let uploadedThumbnail = req.compressedFiles.find(cf => cf.fieldname == "c_thumbnail");

        const slug = slugMaker(req.body.c_slug);

        let data = { 
            ...req.body, 
            c_status: "draft", 
            c_slug: slug, 
            c_tags: JSON.stringify(JSON.parse(req.body.c_tags || [])),
            c_author: req.user.uid
        };

        delete data.categories;


        if(req.theMode == "submission"){
            if(!uploadedThumbnail) {
                const file = await fs.promises.readFile(path.join(__basedir, "public", submissionPost.u_thumbnail_link));
                if(file) uploadedThumbnail = { buffer: file };
            };
            if(!uploadedFile && submissionPost.u_video_link){
                const file = await fs.promises.readFile(path.join(__basedir, "public", submissionPost.u_video_link));
                
                if(file) uploadedFile = { buffer: file };
            }
        }
        
        const { insertId } = await createNewPost(data);
    
        const categoryIds = JSON.parse(req.body.categories);
        await createPostCategoryPivot(insertId, categoryIds);


        const returnObj = { status: true, message: "Post uploaded", data: { c_id: insertId, ...data, c_status: req.body.c_status }};

        if(req.theMode != "submission") res.json(returnObj)


        //Do The Thumbnail
        if(uploadedThumbnail){

            const filename = `${slug}.webp`;
            const filePath = path.join(__basedir, 'public/files/images', filename);
            
            const [ settings, _] = await Promise.all([
                getWebSettings(),
                fs.promises.writeFile(filePath, uploadedThumbnail.buffer)
            ])
            
            const dataObj = { c_status: (uploadedFile || isURL(req.body.c_download_link)) ? "draft" : "published", c_thumbnail: `/files/images/${filename}` };
            
            if(settings.website_save_to_cloud) {
                const s3Key = `images/${filename}`;
                uploadFileToS3(filePath, s3Key);
                dataObj.c_thumbnail_backup = s3Key
            }

            await editPostById({ postId: insertId, data: dataObj })

        }

        setImmediate(async () => {
            if(uploadedFile || isURL(req.body?.c_download_link)){
    
                const slug = slugMaker(data.c_slug);
                let result = {};

                if(isURL(req.body?.c_download_link)){

                    result = await downloadVideo({
                        title: `${slug}-${generateRandomId()}`,
                        videoUrl: req.body.c_download_link
                    });

                }else{

                    // Save buffer to a temp file
                    const tempInputPath = path.join(__basedir, "public/files/videos", `temp_${generateRandomId()}.mp4`);
                    fs.writeFileSync(tempInputPath, uploadedFile.buffer);
                    result = await processVideo(slug, tempInputPath);
                    
                    // Optional: delete temp input file
                    fs.unlinkSync(tempInputPath);

                }

        
                data = {
                    c_download_link: result.path, 
                    c_download_id: result.slug, 
                    c_teaser_link: result.teaser,
                    c_resolution: result?.info?.resolution,
                    c_quality: result?.info?.quality,
                    c_duration: result?.info?.duration,
                    c_status: req.body.c_status
                };
        
                editPostById({ data, postId: insertId })
            }
        });

        if(req.theMode == "submission") return returnObj
      

    } catch (error) {
        let message;

        if (error.sqlMessage?.includes("c_link")) {
            message = "Post already exists"
        } else message = error?.sqlMessage || "Something went wrong";

        //Response To Client
        return req.theMode == "submission" ? { status: false, message } : res.json({ status: false, message })

    };
      
})

//EDIT POST
exports.adminPostPut = asyncHandler(async (req, res, next) => {

    const post = await getPostById(req.query.id);
    if(!post) return res.json({ status: false, message: "Post not found" });
    
    const settings = await getWebSettings();
        
    const filterResult = filterPostData({ title: req.body.c_title, description: req.body.c_description, tags: JSON.parse(req.body.c_tags || []).join(" ") }, JSON.parse(settings.website_banned_words));
    if(!filterResult.status) {
        return res.json(filterResult);
    }

    const slug = slugMaker(req.body.c_slug);
    let data = { c_id: req.query.id, ...req.body, c_slug: slug, c_tags: JSON.stringify(JSON.parse(req.body.c_tags || [])) };

    const uploadedFile = req.compressedFiles.find(cf => cf.fieldname == "c_file");
    const uploadedThumbnail = req.compressedFiles.find(cf => cf.fieldname == "c_thumbnail");

    //Do The Thumbnail
    if(uploadedThumbnail){

        const filename = `${slug}.webp`;
        const filePath = path.join(__basedir, 'public/files/images', filename);
        
       await fs.promises.writeFile(filePath, uploadedThumbnail.buffer)
        
        data.c_thumbnail = `/files/images/${filename}`;

        //Saving To Cloud
        if(settings.website_save_to_cloud){
            const s3Key = `images/${filename}`;
            uploadFileToS3(filePath, s3Key);

            data.c_thumbnail_backup = s3Key;
        }

        if(post.c_thumbnail != data.c_thumbnail) {
            Promise.allSettled([
                post.c_thumbnail && deleteFileAttachmentMulter(path.join(__basedir, 'public', post.c_thumbnail)),
                post.c_thumbnail_backup && deleteS3File(process.env.AWS_BUCKET_NAME, post.c_thumbnail_backup)
            ])
        };
        

    }

    //Update Category
    const currentCatIds = (await getCategoriesOfPost(req.query.id)).map(cat => cat.cc_id);
    const postCats = JSON.parse(req.body.categories);

    // Categories to ADD (in `postCats` but not in `currentCatIds`)
    const categoriesToAdd = postCats.filter(catId => !currentCatIds.includes(catId));

    // Categories to DELETE (in `currentCatIds` but not in `postCats`)
    const categoriesToDelete = currentCatIds.filter(catId => !postCats.includes(catId));

    delete data.categories;

    await Promise.all([
        createPostCategoryPivot(req.query.id, categoriesToAdd),
        deletePostCategoryPivot(categoriesToDelete),
        editPostById({ data, postId: req.query.id })
    ]);

    res.json({ status: true, message: "Changes Saved", data });

    if(uploadedFile || isURL(req.body?.c_download_link)){

        const slug = slugMaker(data.c_slug);

        let result = {};

        editPostById({ data: { c_status: "draft" }, postId: req.query.id })

        if(isURL(req.body?.c_download_link)){

            result = await downloadVideo({
                title: `${slug}-${generateRandomId()}`,
                videoUrl: req.body.c_download_link
            });

        }else{

            // Save buffer to a temp file
            const tempInputPath = path.join(__basedir, "public/files/videos", `temp_${generateRandomId()}.mp4`);
            fs.writeFileSync(tempInputPath, uploadedFile.buffer);
        
            result = await processVideo(slug, tempInputPath);

            // Optional: delete temp input file
            fs.unlinkSync(tempInputPath);
        }
        
  
        data = { 
            ...data,
            c_download_link: result.path, 
            c_download_id: result.slug, 
            c_teaser_link: result.teaser,
            c_resolution: result?.info?.resolution,
            c_quality: result?.info?.quality,
            c_duration: result?.info?.duration
        };
        

        if(post.c_download_link != data.c_download_link){

            const folderPath = path.join(__basedir, 'public', post.c_download_link.split("/index.m3u8")[0]);

            deleteFolderRecursive(folderPath, (err) => {
                if (err) {
                    console.error("❌ Failed to delete folder:", err.message);
                } else {
                    console.log("✅ Folder deleted successfully.");
                }
            })

            //Delete From S3
            post.c_thumbnail_backup && deleteS3Folder(`videos/${post.c_download_id}`);
            
        }

        editPostById({ data, postId: req.query.id })

    }
});

//DELETE POST
exports.adminPostDelete = asyncHandler(async (req, res, next) => {
    
    const { deletedRows, deletedData } = await deletePosts(req.body)

    let message = "";
    if(deletedRows > 1) message = `${deletedRows} Posts Deleted`
    else if(deletedRows == 1) message = "Post Deleted";
    else message = "No Post Deleted"

    res.json({ status: Boolean(deletedRows), message });

    const deleteTasks = [];

    deletedData.forEach(row => {
        if (row.c_thumbnail) {
            Promise.allSettled([
                deleteTasks.push(deleteFileAttachmentMulter(path.join(__basedir, 'public', row.c_thumbnail))),
                deleteS3File(process.env.AWS_BUCKET_NAME, row.c_thumbnail_backup)
            ])
        };

        if (row.c_download_link) {

            const folderPath = path.join(__basedir, 'public', row.c_download_link.split("/index.m3u8")[0]);

            deleteFolderRecursive(folderPath, (err) => {
                if (err) {
                    console.error("❌ Failed to delete folder:", err.message);
                } else {
                    console.log("✅ Folder deleted successfully.");
                }
            })

            //Delete From S3 (If There Is Thumbnail Backup, That Means Videos Folder Was Uploaded)
            row.c_thumbnail_backup && deleteS3Folder(`videos/${row.c_download_id}`);
        };
    });

    Promise.allSettled(deleteTasks)

});
