const fs = require("fs");
const path = require("path");
const asyncHandler = require("../../../helpers/asyncHandler");
const { slugMaker } = require("../../../helpers/dataManipulator");
const { getNextOffset, getPaginationNextPrev } = require("../../../helpers/pagination");
const { getPostCategories, createPostCategory, editPostCategory, deletePostCategories, getPostCategoryById } = require("../../../helpers/post");
const { getWebSettings } = require("../../../helpers/settings");
const { uploadFileToS3, deleteS3File } = require("../../../helpers/s3");
const { deleteFileAttachmentMulter } = require("../../../multer/multerMiddleware");


//ADMIN POST CATEGORIES
exports.adminGetPostCategories = asyncHandler(async (req, res, next) => {
    
    const limit = req.query.limit || parseInt(process.env.LIMIT), currentPage = parseInt(req.query.page) || 1, search = req.query.search;
    
    const { totalCount, categories } = await getPostCategories({
        query: { search, limit, offset: getNextOffset(currentPage, limit) }
    });

    const { prevPage, nextPage, totalPages } = getPaginationNextPrev({ page: currentPage, limit, totalCount});


    res.json({ status: true, data: { categories, pagination: { currentPage, prevPage, nextPage, totalPages } }})

});

//ADMIN CATEGORY (POST)
exports.adminPostCategoryPost = asyncHandler(async (req, res, next) => {
    
    try {

        req.body.cc_slug = slugMaker(req.body.cc_slug);

        let uploadedThumbnail = req.compressedImgs.find(cf => cf.fieldname == "cc_thumbnail");
        
        if(uploadedThumbnail) {
            const filename = `${req.body.cc_slug}.webp`;
            const filePath = path.join(__basedir, 'public/files/categories-imgs', filename);
            await fs.promises.writeFile(filePath, uploadedThumbnail.buffer);
            req.body.cc_thumbnail = `/files/categories-imgs/${filename}`

            const { website_save_to_cloud } = await getWebSettings();

            if(website_save_to_cloud) {
                const s3Key = `images/categories-imgs/${filename}`;
                uploadFileToS3(filePath, s3Key);
                req.body.cc_thumbnail_backup = s3Key
            }
        }

        
        const { insertId } = await createPostCategory(req.body);
        req.body.cc_id = insertId;
        req.body.cc_created_at = new Date();
        

        return res.json({ status: true, message: "Category created", data: req.body })

    } catch (error) {
        let message;

        if (error.sqlMessage?.includes("name") || error.sqlMessage?.includes("slug")) {
            message = "Category already exists"
        } else message = error?.sqlMessage || "Something went wrong";

        //Response To Client
        return res.json({ status: false, message })

    };

});

//ADMIN CATEGORY (PUT)
exports.adminPostCategoryPut = asyncHandler(async (req, res, next) => {

    const category = await getPostCategoryById(req.query.id);
    if(!category) return res.json({ status: false, message: "Category not found" });
    
    if(req.body.cc_slug) req.body.cc_slug = slugMaker(req.body.cc_slug);

    delete req.body.postCount;
    
    let uploadedThumbnail = req.compressedImgs.find(cf => cf.fieldname == "cc_thumbnail");
        
    if(uploadedThumbnail) {
        const filename = `${req.body.cc_slug}.webp`;
        const filePath = path.join(__basedir, 'public/files/categories-imgs', filename);
        await fs.promises.writeFile(filePath, uploadedThumbnail.buffer);
        req.body.cc_thumbnail = `/files/categories-imgs/${filename}`

        const { website_save_to_cloud } = await getWebSettings();

        if(website_save_to_cloud) {
            const s3Key = `images/categories-imgs/${filename}`;
            uploadFileToS3(filePath, s3Key);
            req.body.cc_thumbnail_backup = s3Key
        }

        if(category.cc_thumbnail != req.body.cc_thumbnail) {
            category.cc_thumbnail && deleteFileAttachmentMulter(path.join(__basedir, 'public', category.cc_thumbnail));
            category.cc_thumbnail_backup && deleteS3File(process.env.AWS_BUCKET_NAME, category.cc_thumbnail_backup);
        };
    }

    await editPostCategory({ data: req.body, categoryId: req.query.id });
    
    return res.json({ status: true, message: "Category updated", data: req.body })
});

//ADMIN CATEGORY (POST)
exports.adminPostCategoryDelete = asyncHandler(async (req, res, next) => {
    
    const { deletedRows, deletedData } = await deletePostCategories(req.body)

    let message = "";
    if(deletedRows > 1) message = `${deletedRows} Categories Deleted`
    else if(deletedRows == 1) message = "Category Deleted";
    else message = "No Category Deleted"

    res.json({ status: Boolean(deletedRows), message });

    const deleteTasks = [];

    deletedData.forEach(row => {
        if (row.cc_thumbnail) {
            deleteTasks.push(deleteFileAttachmentMulter(path.join(__basedir, 'public', row.cc_thumbnail)));
            deleteS3File(process.env.AWS_BUCKET_NAME, row.cc_thumbnail_backup);
        };
    });

    Promise.allSettled(deleteTasks)
});
