const convert = require('xml-js');
const zlib = require('zlib');
const asyncHandler = require('../../../helpers/asyncHandler');
const { getContentSitemapPagesWithLastMod, getCategorySitemapPagesWithLastMod, getTagSitemapPagesWithLastMod, getAllSitemapLastmods } = require('../../../helpers/sitemap');
const { getWebSettings } = require('../../../helpers/settings');
const { getPublishedContentsWithCategories } = require('../../../helpers/post');
const { getNextOffset } = require('../../../helpers/pagination');
const { VIEW_MULTIPLIER } = require('../../../extras/constants');
const { getPageFunction } = require('../../../helpers/page');


exports.sitemapIndex = asyncHandler(async (req, res, next) => {
    
    let URL;

    const [settings, lastmods] = await Promise.all([
        getWebSettings(),
        getAllSitemapLastmods()
    ]);

    URL = settings.website_url;

    const sitemapTypes = [ 
        { type: 'videos', lastmod: lastmods.videos.toISOString().split('T')[0] },
        { type: 'categories', lastmod: lastmods.categories.toISOString().split('T')[0] },
        { type: 'tags', lastmod: lastmods.tags.toISOString().split('T')[0] },
        { type: 'pages', lastmod: lastmods.pages.toISOString().split('T')[0] }
    ];

    const sitemapIndex = {
        _declaration: {
            _attributes: { version: '1.0', encoding: 'UTF-8' },
        },
        sitemapindex: {
            _attributes: { xmlns: 'http://www.sitemaps.org/schemas/sitemap/0.9' },
            sitemap: sitemapTypes.map(item => ({
                loc: { _text: `${URL}/sitemap-${item.type}.xml` },
                lastmod: { _text: item.lastmod },
            })),
        },
    };

    const xml = convert.js2xml(sitemapIndex, { compact: true, spaces: 2 });

    // Compress using gzip
    zlib.gzip(xml, (err, zippedXml) => {
        if (err) {
            return next(err);
        }

        res.setHeader('Content-Type', 'application/xml');
        res.setHeader('Content-Encoding', 'gzip');
        res.setHeader('Content-Disposition', 'inline; filename="sitemap.xml.gz"');
        
        res.send(zippedXml);
    });

    
})

//GENERATE SITEMAP PAGE
exports.sitemapPage = asyncHandler(async (req, res, next) => {
    
    let URL, type = req.params.type, pages = [], sitemapIndex = {};
    
    const settings = await getWebSettings();
    URL = settings.website_url;

    if(type == "videos") {

        pages = await getContentSitemapPagesWithLastMod();

        sitemapIndex = {
            _declaration: {
                _attributes: { version: '1.0', encoding: 'UTF-8' },
            },
            sitemapindex: {
                _attributes: { xmlns: 'http://www.sitemaps.org/schemas/sitemap/0.9' },
                sitemap: pages.map(item => ({
                    loc: { _text: `${URL}/sitemaps/videos-${item.page}.xml` },
                    lastmod: { _text: item.lastmod },
                })),
            },
        };

    }
    else if(type == "categories") {

        const pages = await getCategorySitemapPagesWithLastMod();
        
        sitemapIndex = {
            _declaration: {
                _attributes: { version: '1.0', encoding: 'UTF-8' },
            },
            urlset: {
                _attributes: {
                xmlns: 'http://www.sitemaps.org/schemas/sitemap/0.9',
                'xmlns:image': 'http://www.google.com/schemas/sitemap-image/1.1',
                },
                url: pages.map(item => ({
                    loc: { _text: `${URL}/category/${item.cc_slug}` },
                    lastmod: { _text: item.lastmod },
                    changefreq: { _text: 'daily' },
                    priority: { _text: '0.8' },
                    'image:image': {
                        'image:loc': { _text: item.thumbnail ? `${URL}${item.thumbnail}` : `${URL}${settings.website_favicon}` },
                        'image:title': { _text: item.cc_name },
                        'image:caption': { _text: item.cc_description || `${item.cc_name} videos on ${settings.website_title}` },
                    },
                })),
            },
        };

    }
    else if(type == "pages") {

        const pages = await getPageFunction(1000, 0);
        
        sitemapIndex = {
            _declaration: {
                _attributes: { version: '1.0', encoding: 'UTF-8' },
            },
            urlset: {
                _attributes: {
                    xmlns: 'http://www.sitemaps.org/schemas/sitemap/0.9'
                },
                url: pages.map(item => ({
                    loc: { _text: `${URL}/page/${item.page_slug}` },
                    lastmod: { _text: new Date(item.page_updated_at).toISOString().split('T')[0] },
                    changefreq: { _text: 'monthly' },
                    priority: { _text: '0.5' }
                })),
            },
        };

        //Add The Homepage To First
        sitemapIndex.urlset.url.unshift(
            {
                loc: { _text: `${URL}/` },
                lastmod: { _text: new Date().toISOString().split('T')[0] },
                changefreq: { _text: 'daily' },
                priority: { _text: '1.0' },
            }
        );

    }
    else if(type == "tags") {

        const pages = await getTagSitemapPagesWithLastMod();
        
        sitemapIndex = {
            _declaration: {
                _attributes: { version: '1.0', encoding: 'UTF-8' },
            },
            urlset: {
                _attributes: {
                    xmlns: 'http://www.sitemaps.org/schemas/sitemap/0.9'
                },
                url: pages.map(item => ({
                    loc: { _text: `${URL}/tag/${encodeURIComponent(item.tag)}` },
                    lastmod: { _text: item.lastmod },
                    changefreq: { _text: 'weekly' },
                    priority: { _text: '0.6' }
                })),
            },
        };

    }

    const xml = convert.js2xml(sitemapIndex, { compact: true, spaces: 2 });

    // Compress using gzip
    zlib.gzip(xml, (err, zippedXml) => {
        if (err) {
            return next(err);
        }

        res.setHeader('Content-Type', 'application/xml');
        res.setHeader('Content-Encoding', 'gzip');
        res.setHeader('Content-Disposition', 'inline; filename="sitemap.xml.gz"');
        
        res.send(zippedXml);
    });
    
})

//GENERATE SITEMAP PAGE CONTENT
exports.sitemapPageContent = asyncHandler(async (req, res, next) => {
    
    const CONTENT_LIMIT = 50000;

    let URL, type = req.params.type, currentPage = Number(req.params.page), sitemapObj = {};
    
    const settings = await getWebSettings();
    URL = settings.website_url;

    if(type == "videos") {

        
        const [settings, pages] = await Promise.all([
            getWebSettings(),
            getPublishedContentsWithCategories(CONTENT_LIMIT, getNextOffset(currentPage, CONTENT_LIMIT))
        ]);
        
        sitemapObj = {
            _declaration: {
            _attributes: {
                version: '1.0',
                encoding: 'UTF-8',
            },
            },
            urlset: {
            _attributes: {
                xmlns: 'http://www.sitemaps.org/schemas/sitemap/0.9',
                'xmlns:video': 'http://www.google.com/schemas/sitemap-video/1.1',
            },
            url: pages.map(item => {
                const pubDate = new Date(item.c_created_at).toISOString();
                const lastmod = new Date(item.c_updated_at).toISOString().split('T')[0];
                const tags = Array.isArray(item.c_tags) ? item.c_tags : JSON.parse(item.c_tags || '[]');
                const categories = JSON.parse(item.categories || '[]');

                return {
                    loc: { _text: `${URL}/${item.c_slug}` },
                    lastmod: { _text: lastmod },
                    changefreq: { _text: 'weekly' },
                    priority: { _text: '0.8' },
                    ...(item.c_teaser_link && {
                        'video:video': {
                        'video:thumbnail_loc': { _text: `${URL}${item.c_thumbnail}` },
                        'video:title': { _cdata: item.c_title },
                        'video:description': {
                            _cdata:
                            item.c_description?.replace(/(<([^>]+)>)/gi, '') ||
                            `${item.c_title} on ${settings.website_title}`,
                        },
                        'video:content_loc': { _text: `${URL}${item.c_teaser_link}` },
                        'video:duration': { _text: String(item.c_duration || 0) },
                        'video:view_count': { _text: String(item.c_views * VIEW_MULTIPLIER || 0) },
                        'video:family_friendly': { _text: 'no' },
                        'video:rating': { _text: 'adult' },
                        'video:publication_date': { _text: pubDate },
                        'video:category': {
                            _cdata: categories[0]?.cc_name || 'Uncategorized',
                        },
                        ...(tags.length > 0
                            ? tags.reduce((acc, tag) => {
                                acc['video:tag'] = acc['video:tag'] || [];
                                acc['video:tag'].push({ _cdata: tag });
                                return acc;
                            }, {})
                            : {}),
                        },
                    }),
                };

            }),
            },
        };

    };

    const xml = convert.js2xml(sitemapObj, { compact: true, spaces: 2 });

    // Compress using gzip
    zlib.gzip(xml, (err, zippedXml) => {
        if (err) {
            return next(err);
        }

        res.setHeader('Content-Type', 'application/xml');
        res.setHeader('Content-Encoding', 'gzip');
        res.setHeader('Content-Disposition', 'inline; filename="sitemap.xml.gz"');
        
        res.send(zippedXml);
    });
    
})