const log4js = require("log4js");
const fs = require("fs");
const path = require("path");
const db = require("../models/db");

let isDbAvailable = false;
let dbCheckAttempted = false;

// Extract stack from error or logger
function getStackInfoFromMessage(message, fallbackStackIndex = 5) {
  let stack = (message instanceof Error && message.stack) || null;

  if (stack) {
    const stackLines = stack.split('\n');
    for (let i = 1; i < stackLines.length; i++) {
      const match = stackLines[i].match(/\s+at\s+(.*):(\d+):(\d+)/);
      if (match) {
        const fullPath = match[1].trim();
        const filePath = path.relative(process.cwd(), fullPath);
        if (!filePath.includes('node_modules') && !filePath.includes('logger.js')) {
          return {
            filePath,
            lineNumber: match[2],
            columnNumber: match[3]
          };
        }
      }
    }
  }

  // Fallback if no valid stack found
  const fallback = new Error().stack.split('\n')[fallbackStackIndex] || '';
  const fallbackMatch = fallback.match(/\s+at\s+(.*):(\d+):(\d+)/);
  return fallbackMatch
    ? {
        filePath: path.relative(process.cwd(), fallbackMatch[1].trim()),
        lineNumber: fallbackMatch[2],
        columnNumber: fallbackMatch[3]
      }
    : { filePath: 'unknown', lineNumber: 0, columnNumber: 0 };
}

// Format message
function formatLogMessage(message) {
  const result = {
    status: 200,
    text: '',
    stack: null,
    context: null
  };

  if (message instanceof Error) {
    result.status = 500;
    result.text = message.message;
    result.stack = message.stack;
    result.context = {
      name: message.name,
      ...(message.code && { code: message.code })
    };
  } else if (typeof message === 'object' && message !== null) {
    result.status = message.status || 500;
    result.text = message.message || JSON.stringify(message);
    result.context = { ...message };
    delete result.context.message;
  } else {
    result.text = String(message);
  }

  return result;
}

// Layout
log4js.addLayout('enhanced', function (config) {
  return function (logEvent) {
    const message = formatLogMessage(logEvent.data[0]);
    const stackInfo = getStackInfoFromMessage(logEvent.data[0]);

    return JSON.stringify({
      timestamp: logEvent.startTime.toISOString(),
      level: logEvent.level.levelStr,
      status: message.status,
      message: message.text,
      filePath: stackInfo.filePath,
      lineNumber: stackInfo.lineNumber,
      fullStack: message.stack || null,
      context: message.context
    });
  };
});

// Configure
log4js.configure({
  appenders: {
    fileAppender: {
      type: 'file',
      filename: 'debug-log.log',  // Absolute path for the log file
      compress: true,
      maxLogSize: 10 * 1024 * 1024,
      backups: 5,
      layout: {
        type: 'pattern',
        pattern: '[%d] [%p] [%c] %f:%l - %m\n\n'
      }
    }
  },
  categories: {
    default: {
      appenders: ['fileAppender'],
      level: 'debug' // Log everything from debug and up
    },
    db: {
      appenders: ['fileAppender'],
      level: 'debug'
    }
  }
});

const fileLogger = log4js.getLogger();

function tryDbLog(level, message, callback) {
  if (!dbCheckAttempted) {
    db.query("SELECT 1", (err) => {
      isDbAvailable = !err;
      dbCheckAttempted = true;
      if (err) fileLogger.warn(`Database logging unavailable: ${err.message}`);
      attemptDbLog(level, message, callback);
    });
  } else {
    attemptDbLog(level, message, callback);
  }
}

function attemptDbLog(level, message, callback) {
  if (isDbAvailable) {
    const formatted = formatLogMessage(message);
    const stackInfo = getStackInfoFromMessage(message);

    db.query(
      `INSERT INTO f_logs 
       (l_type, l_status, l_message, l_file_path, l_line_number, l_stack, l_context) 
       VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [
        level,
        formatted.status,
        formatted.text,
        stackInfo.filePath,
        stackInfo.lineNumber,
        formatted.stack,
        formatted.context ? JSON.stringify(formatted.context) : null
      ],
      (err) => {
        if (err) {
          fileLogger.error(`Failed to write log to database: ${err.message}`);
          callback(false);
        } else {
          fileLogger.info({
            level,
            status: formatted.status,
            message: formatted.text,
            filePath: stackInfo.filePath,
            lineNumber: stackInfo.lineNumber,
            stack: formatted.stack,
            context: formatted.context
          });
          callback(true);
        }
      }
    );
  } else {
    callback(false);
  }
}

const logger = {
  debug: (message, callback) => {
    tryDbLog('debug', message, (dbLogged) => {
      if (!dbLogged) fileLogger.debug(message);
      if (callback) callback();
    });
  },
  info: (message, callback) => {
    tryDbLog('info', message, (dbLogged) => {
      if (!dbLogged) fileLogger.info(message);
      if (callback) callback();
    });
  },
  warn: (message, callback) => {
    tryDbLog('warning', message, (dbLogged) => {
      if (!dbLogged) fileLogger.warn(message);
      if (callback) callback();
    });
  },
  error: (message, callback) => {
    tryDbLog('error', message, (dbLogged) => {
      if (!dbLogged) fileLogger.error(message);
      if (callback) callback();
    });
  },
  emergency: (message) => {
    try {
      const timestamp = new Date().toISOString();
      let logText = `[${timestamp}] [EMERGENCY] `;
  
      // Format the message and stack info if it's an Error object
      const formatted = formatLogMessage(message);
      const stackInfo = getStackInfoFromMessage(message);
  
      logText += `${formatted.text}\n`;
      logText += `Status: ${formatted.status}\n`;
      logText += `Location: ${stackInfo.filePath}:${stackInfo.lineNumber}\n`;
      logText += `Stack Trace:\n${formatted.stack || 'No stack trace available'}\n\n`;

      // Write to fallback log file
      fs.appendFileSync(path.join(process.cwd(), "debug-log.log"), logText, "utf8");
  
      // Log to console and use fileLogger if available
      //console.error(logText);
      if (fileLogger?.fatal) {
        fileLogger.fatal(formatted);
        fileLogger.fatal(logText);
      }
    } catch (err) {
      // Final fallback if writing even fails
      //console.error("CRITICAL: Failed to write emergency log:", err);
    }
  }
};

// Initial DB check
db.query("SELECT 1", (err) => {
  isDbAvailable = !err;
  dbCheckAttempted = true;
  if (err) logger.emergency(`Failed to initialize database logging: ${err.message}`);
});

module.exports = logger;
