const { Expo } = require("expo-server-sdk");
const { getUserPreferencesByPushtoken } = require("./user");
const db = require("../models/db");


exports.pushNotification = async (unSortedExpoPushTokens, data) => {
    
    //Send Notification To Only User Who Allowed Push Notification
    if (!unSortedExpoPushTokens || !unSortedExpoPushTokens.length) return;

    const limit = (await import('p-limit')).default(Number(process.env.PROMISE_LIMIT));

    const expoPushTokens = (await Promise.all(
        unSortedExpoPushTokens.map(token =>
          limit(async () => {
            const user = await getUserPreferencesByPushtoken(token);
            return user?.up_push_notification ? token : null;
          })
        )
    )).filter(token => token !== null);

    const expo = new Expo({ accessToken: process.env.EXPO_ACCESS_TOKEN });

    const chunks = expo.chunkPushNotifications(
        expoPushTokens.map((token) => ({ to: token, ...data }))
    );

    const tickets = [];

    for (const chunk of chunks) {
        try {
            const ticketChunk = await expo.sendPushNotificationsAsync(chunk);
            tickets.push(...ticketChunk);
        } catch (error) {
            console.error(error);
        }
    }

    const responses = [];

    for (const ticket of tickets) {
        if (ticket.status === "error") {
            if (ticket.details && ticket.details.error === "DeviceNotRegistered") {
                responses.push("DeviceNotRegistered");
            }
        }

        if (ticket.status === "ok") {
            responses.push(ticket.id);
        }
    }

    return responses;
};

// GET SYSTEM NOTIFICATIONS
exports.getUserSystemNotifications = (obj) => {
    return new Promise(async (resolve, reject) => {
        try {
            // First get the user's creation date
            const userCreatedAtQuery = `SELECT created_at FROM f_users WHERE uid = ${obj.query.userId}`;
            const [userCreatedAtResult] = await new Promise((res, rej) => {
                (obj.connection || db).query(userCreatedAtQuery, (err, data) => err ? rej(err) : res(data));
            });
            
            const userCreatedAt = new Date(userCreatedAtResult?.created_at).toISOString().slice(0, 19).replace('T', ' ');
            
            // Build dynamic filters
            let filters = `WHERE (sn_user_id = ${obj.query.userId} OR (sn_user_id IS NULL AND sn_created_at >= '${userCreatedAt}'))`;

            // Add filter to exclude deleted global notifications
            filters += `
                AND (sn.sn_user_id IS NOT NULL 
                OR (sn.sn_user_id IS NULL AND (snt.snt_has_deleted IS NULL OR snt.snt_has_deleted = 0)))
            `;

            // Rest of the query remains the same...
            const query = `SELECT * FROM f_system_notifications sn 
                           LEFT JOIN f_system_notifications_track snt 
                           ON sn.sn_id = snt.snt_notification_id AND snt.snt_user_id = ${obj.query.userId} 
                           ${filters} 
                           ORDER BY sn.sn_id DESC 
                           LIMIT ${obj.query.limit} OFFSET ${obj.query.offset};`;

            const countQuery = `SELECT COUNT(*) as totalCount 
                                FROM f_system_notifications sn 
                                LEFT JOIN f_system_notifications_track snt 
                                ON sn.sn_id = snt.snt_notification_id AND snt.snt_user_id = ${obj.query.userId} 
                                ${filters};`;

            const dbInstance = obj.connection || db;

            const [dataResult, countResult] = await Promise.all([
                new Promise((res, rej) => dbInstance.query(query, (err, data) => err ? rej(err) : res(data))),
                new Promise((res, rej) => dbInstance.query(countQuery, (err, data) => err ? rej(err) : res(data))),
            ]);

            return resolve({
                notifications: dataResult,
                totalCount: countResult[0].totalCount
            });

        } catch (err) {
            return reject(err);
        }
    });
};


//GET USER SYSTEM NOTIFICATION BY ID
exports.getSystemNotificationById = (notificationId) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_system_notifications WHERE sn_id = ?", [notificationId], (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
}

//GET USER UNREAD SYSTEM NOTIFICATIONS
exports.getUserUnreadSystemNotifications = (userId) => {
    return new Promise(async (resolve, reject) => {
        db.query("SELECT sn.* FROM f_system_notifications sn LEFT JOIN f_system_notifications_track snt ON sn.sn_id = snt.snt_notification_id AND snt.snt_user_id = ? WHERE ( sn.sn_user_id = ? AND sn.sn_has_read = 0 ) OR ( sn.sn_user_id IS NULL AND snt.snt_notification_id IS NULL ) ORDER BY sn.sn_id DESC", [userId, userId], (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
}

//CREATE NEW SYSTEM NOTIFICATION
exports.createSystemNotification = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_system_notifications SET ?", obj, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
}

//CREATE NEW SYSTEM NOTIFICATION TRACK
exports.createSystemNotificationTrack = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_system_notifications_track SET ?", obj, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
}

//UPDATE SYSTEM NOTIFICATION
exports.updateSystemNotification = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("UPDATE f_system_notifications SET ? WHERE sn_user_id = ? AND sn_id = ?", [obj.data, obj.userId, obj.notificationId], (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
}

//MARK SYSTEM NOTIFICATIONS AS READ
exports.markSystemNotificationAsRead = (userId) => {
    return new Promise((resolve, reject) => {
        db.query("UPDATE f_system_notifications SET sn_has_read = 1 WHERE sn_user_id = ?", userId, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
}

exports.markSystemNotificationAsRead = (userId, connection) => {
    return new Promise((resolve, reject) => {
      const conn = connection || db;
  
      // Step 1: Mark personal notifications as read
      conn.query("UPDATE f_system_notifications SET sn_has_read = 1 WHERE sn_user_id = ?", [userId], (err) => {
        if (err) return reject(err);
  
        // Step 2: Find untracked global notifications
        const globalQuery = `
          SELECT sn.sn_id
          FROM f_system_notifications sn
          LEFT JOIN f_system_notifications_track snt
              ON sn.sn_id = snt.snt_notification_id AND snt.snt_user_id = ?
          WHERE sn.sn_user_id IS NULL AND snt.snt_notification_id IS NULL
        `;
  
        conn.query(globalQuery, [userId], (err, globalNotifs) => {
          if (err) return reject(err);
  
          // Step 3: Insert tracking rows if any
          if (globalNotifs.length > 0) {
            const values = globalNotifs.map(notif => [userId, notif.sn_id]);
  
            conn.query(
              "INSERT INTO f_system_notifications_track (snt_user_id, snt_notification_id) VALUES ?",
              [values],
              (err, result) => {
                if (err) return reject(err);
                return resolve({ success: true, inserted: globalNotifs.length });
              }
            );
          } else {
            return resolve({ success: true, inserted: 0 });
          }
        });
      });
    });
};
  
  

//DELETE SYSTEM NOTIFICATION
exports.deleteUserSystemNotification = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("DELETE FROM f_system_notifications WHERE sn_user_id = ? AND sn_id = ?", [obj.userId, obj.notificationId], (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
}

//DELETE ALL USER SYSTEM NOTIFICATIONS
exports.deleteUserSystemNotifications = (userId) => {
    return new Promise((resolve, reject) => {
        db.query("DELETE FROM f_system_notifications WHERE sn_user_id = ?", userId, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
}

exports.deleteAllUserSystemNotifications = (userId, connection) => {
    return new Promise((resolve, reject) => {
        const conn = connection || db;

        // Step 1: Delete all personal notifications for the user
        conn.query("DELETE FROM f_system_notifications WHERE sn_user_id = ?", [userId], (err) => {
            if (err) return reject(err);

            // Step 2: Find untracked global notifications
            const globalQuery = `
                SELECT sn.sn_id
                FROM f_system_notifications sn
                LEFT JOIN f_system_notifications_track snt
                    ON sn.sn_id = snt.snt_notification_id AND snt.snt_user_id = ?
                WHERE sn.sn_user_id IS NULL AND snt.snt_notification_id IS NULL
            `;

            conn.query(globalQuery, [userId], (err, globalNotifs) => {
                if (err) return reject(err);

                // Step 3: Process untracked global notifications
                if (globalNotifs.length > 0) {
                    const values = globalNotifs.map(notif => [userId, notif.sn_id]);

                    // Insert into tracking table for untracked global notifications
                    conn.query(
                        "INSERT INTO f_system_notifications_track (snt_user_id, snt_notification_id) VALUES ?",
                        [values],
                        (err) => {
                            if (err) return reject(err);

                            // After inserting, mark those global notifications as deleted
                            const deleteQuery = `
                                UPDATE f_system_notifications_track 
                                SET snt_has_deleted = 1 
                                WHERE snt_user_id = ? AND snt_notification_id IN (?) 
                            `;
                            conn.query(deleteQuery, [userId, globalNotifs.map(notif => notif.sn_id)], (err) => {
                                if (err) return reject(err);
                                return resolve({ success: true, deleted: globalNotifs.length });
                            });
                        }
                    );
                };

                const updateQuery = `
                        UPDATE f_system_notifications_track 
                        SET snt_has_deleted = 1 
                        WHERE snt_user_id = ?
                    `;
                conn.query(updateQuery, [userId], (err) => {
                    if (err) return reject(err);
                    return resolve({ success: true, deleted: 0 });
                });

            });
        });
    });
};
