const db = require("../models/db");

//GET USER METRICS GROWTH
exports.getUsersForDashboard = () => {
    return new Promise((resolve, reject) => {
        // Get current and previous month dates
        const currentDate = new Date();
        const currentMonth = currentDate.getMonth() + 1;
        const currentYear = currentDate.getFullYear();
        
        const lastMonthDate = new Date();
        lastMonthDate.setMonth(lastMonthDate.getMonth() - 1);
        const lastMonth = lastMonthDate.getMonth() + 1;
        const lastMonthYear = lastMonthDate.getFullYear();

        // Run all queries simultaneously
        Promise.all([
            // Query for this month's counts by type
            new Promise((res, rej) => {
                db.query(`
                    SELECT ue.ue_user_type AS type, COUNT(*) AS count 
                    FROM f_users u
                    JOIN f_users_essentials ue ON u.uid = ue.ue_user_id
                    WHERE MONTH(u.created_at) = ? AND YEAR(u.created_at) = ?
                    GROUP BY ue.ue_user_type
                `, [currentMonth, currentYear], (err, data) => {
                    if (err) rej(err);
                    else res(data);
                });
            }),
            
            // Query for last month's counts by type
            new Promise((res, rej) => {
                db.query(`
                    SELECT ue.ue_user_type AS type, COUNT(*) AS count 
                    FROM f_users u
                    JOIN f_users_essentials ue ON u.uid = ue.ue_user_id
                    WHERE MONTH(u.created_at) = ? AND YEAR(u.created_at) = ?
                    GROUP BY ue.ue_user_type
                `, [lastMonth, lastMonthYear], (err, data) => {
                    if (err) rej(err);
                    else res(data);
                });
            })
        ])
        .then(([thisMonthData, lastMonthData]) => {
            // Initialize result structure
            const result = {
                regular: { thisMonth: 0, lastMonth: 0 },
                agent: { thisMonth: 0, lastMonth: 0 },
                api: { thisMonth: 0, lastMonth: 0 }
            };

            // Process this month's data
            thisMonthData.forEach(row => {
                result[row.type].thisMonth = row.count;
            });

            // Process last month's data
            lastMonthData.forEach(row => {
                result[row.type].lastMonth = row.count;
            });

            resolve(result);
        })
        .catch(err => {
            reject(err);
        });
    });
};

exports.getUsersCount = () => {
    return new Promise((resolve, reject) => {
       
        Promise.all([
            // Query for this month's counts by type
            new Promise((res, rej) => {
                db.query(`SELECT ue.ue_user_type AS type, COUNT(*) AS count FROM f_users u JOIN f_users_essentials ue ON u.uid = ue.ue_user_id GROUP BY ue.ue_user_type`, (err, data) => {
                    if (err) rej(err);
                    else res(data);
                });
            })
        ])
        .then(([data]) => {
            // Initialize result structure
            const result = {
                regular: 0,
                agent: 0,
                api: 0
            };

            // Process this month's data
            data.forEach(row => {
                result[row.type] = row.count;
            });

            resolve(result);
        })
        .catch(err => {
            reject(err);
        });
    });
};

//GET DAILY TRANSACTION GROWTH METRICS
exports.getDailyTransactionStats = () => {
    return new Promise((resolve, reject) => {
      db.query(`
        SELECT 
          COUNT(CASE WHEN DATE(st_created_at) = CURDATE() THEN 1 END) AS today_count,
          SUM(CASE WHEN DATE(st_created_at) = CURDATE() THEN st_final_amount ELSE 0 END) AS today_amount,
          
          COUNT(CASE WHEN DATE(st_created_at) = DATE_SUB(CURDATE(), INTERVAL 1 DAY) THEN 1 END) AS yesterday_count,
          SUM(CASE WHEN DATE(st_created_at) = DATE_SUB(CURDATE(), INTERVAL 1 DAY) THEN st_final_amount ELSE 0 END) AS yesterday_amount
        FROM f_services_transactions
        WHERE st_status = 'successful'
      `, (err, results) => {
        if (err) return reject(err);
        
        const stats = {
          today: {
            count: results[0].today_count || 0,
            amount: results[0].today_amount || 0
          },
          yesterday: {
            count: results[0].yesterday_count || 0,
            amount: results[0].yesterday_amount || 0
          }
        };
        
        resolve(stats);
      });
    });
};

//GET YEARLY TRANSACTIONS
exports.getYearlyTransactionDataForChart = (year = null) => {
    return new Promise((resolve, reject) => {
        // Handle "all-time" request
        if (year === 'all-time') {
            db.query(`
                SELECT 
                    MONTH(st_created_at) AS month,
                    COUNT(*) AS count,
                    SUM(st_final_amount) AS amount
                FROM f_services_transactions
                WHERE st_status = 'successful'
                AND st_final_amount > 0
                GROUP BY MONTH(st_created_at)
                ORDER BY month ASC
            `, (err, results) => {
                if (err) return reject(err);
                
                const transactionCounts = Array(12).fill(0);
                const transactionAmounts = Array(12).fill(0);
                
                results.forEach(row => {
                    const monthIndex = row.month - 1;
                    transactionCounts[monthIndex] = row.count;
                    transactionAmounts[monthIndex] = parseFloat(row.amount);
                });
                
                resolve({
                    transactionCounts,
                    transactionAmounts,
                    isAllTime: true  // Flag to indicate all-time data
                });
            });
        } else {
            // Handle specific year request
            const targetYear = year || new Date().getFullYear();
            
            db.query(`
                SELECT 
                    MONTH(st_created_at) AS month,
                    COUNT(*) AS count,
                    SUM(st_final_amount) AS amount
                FROM f_services_transactions
                WHERE YEAR(st_created_at) = ?
                AND st_status = 'successful'
                AND st_final_amount > 0
                GROUP BY MONTH(st_created_at)
                ORDER BY month ASC
            `, [targetYear], (err, results) => {
                if (err) return reject(err);
                
                const transactionCounts = Array(12).fill(0);
                const transactionAmounts = Array(12).fill(0);
                
                results.forEach(row => {
                    const monthIndex = row.month - 1;
                    transactionCounts[monthIndex] = row.count;
                    transactionAmounts[monthIndex] = parseFloat(row.amount);
                });
                
                resolve({
                    transactionCounts,
                    transactionAmounts,
                    year: targetYear  // Include the year in response
                });
            });
        }
    });
};

//GET MONTHLY TRANSACTIONS BY TYPE
exports.getMonthlyTransactionsByType = (month = null) => {
    return new Promise((resolve, reject) => {
        const currentDate = new Date();
        let query, params;

        if (month === 'all-time') {
            // Query for all-time data (no month/year filter)
            query = `
                SELECT 
                    st_type,
                    COUNT(*) AS count,
                    SUM(st_final_amount) AS amount
                FROM f_services_transactions
                WHERE st_status = 'successful'
                AND st_final_amount > 0
                GROUP BY st_type
                ORDER BY count DESC`;
            params = [];
        } else {
            // Query for specific month
            const targetMonth = month !== null ? month : currentDate.getMonth() + 1;
            const targetYear = currentDate.getFullYear();
            
            query = `
                SELECT 
                    st_type,
                    COUNT(*) AS count,
                    SUM(st_final_amount) AS amount
                FROM f_services_transactions
                WHERE MONTH(st_created_at) = ?
                AND YEAR(st_created_at) = ?
                AND st_status = 'successful'
                AND st_final_amount > 0
                GROUP BY st_type
                ORDER BY count DESC`;
            params = [targetMonth, targetYear];
        }

        db.query(query, params, (err, results) => {
            if (err) return reject(err);
            
            const transactionTypes = {
                'airtime': 0,
                'data': 0,
                'electricity': 0,
                'cabletv': 0,
                'giftcard': 0,
                'withdrawal': 0,
                'others': 0
            };
            
            results.forEach(row => {
                const type = row.st_type.toLowerCase();
                if (transactionTypes.hasOwnProperty(type)) {
                    transactionTypes[type] = row.amount;
                } else {
                    transactionTypes.others += row.amount;
                }
            });
            
            const chartData = {
                series: [
                    transactionTypes.airtime,
                    transactionTypes.data,
                    transactionTypes.electricity,
                    transactionTypes.cabletv,
                    transactionTypes.giftcard,
                    transactionTypes.withdrawal,
                    transactionTypes.others
                ],
                labels: ['Airtime', 'Data', 'Electricity', 'Cable TV', 'Gift Card', 'Withdrawal', 'Others'],
                colors: ['#7367F0', '#28C76F', '#FF9F43', '#EA5455', '#949494', '#0e1a3e', '#00CFE8'],
                meta: {
                    period: month === 'all-time' ? 'All Time' : 
                          `${new Date(currentDate.getFullYear(), (month || currentDate.getMonth() + 1) - 1).toLocaleString('default', { month: 'long' })} ${currentDate.getFullYear()}`
                }
            };
            
            resolve(chartData);
        });
    });
};

//GET MONTHLY TRANSACTIONS BY USER TYPE
exports.getMonthlyTransactionAmountsByUserType = (month = null) => {
    return new Promise((resolve, reject) => {
        // Validate month input
        if (month !== null && (month < 1 || month > 12)) {
            return reject(new Error('Month must be between 1-12 or "all-time"'));
        }

        let query = `
            SELECT 
                ue.ue_user_type AS user_type,
                SUM(st.st_final_amount) AS total_amount
            FROM f_services_transactions st
            JOIN f_users_essentials ue ON st.st_user_id = ue.ue_user_id
            WHERE st.st_status = 'successful'
            AND st.st_final_amount > 0
        `;

        const params = [];
        const currentDate = new Date();
        
        // Period filtering
        if (month === 'all-time') {
            // No date filter needed
        } else {
            const targetMonth = month !== null ? month : currentDate.getMonth() + 1;
            const targetYear = currentDate.getFullYear();
            
            query += ` AND MONTH(st.st_created_at) = ? AND YEAR(st.st_created_at) = ?`;
            params.push(targetMonth, targetYear);
        }

        query += ` GROUP BY ue.ue_user_type ORDER BY total_amount DESC`;

        db.query(query, params, (err, results) => {
            if (err) return reject(err);
            
            // Initialize with all user types
            const amounts = {
                'regular': 0,
                'agent': 0,
                'api': 0
            };
            
            // Process results
            results.forEach(row => {
                amounts[row.user_type] = parseFloat(row.total_amount) || 0;
            });
            
            // Get month name for display
            let periodDisplay;
            if (month === 'all-time') {
                periodDisplay = 'All Time';
            } else {
                const displayMonth = month || currentDate.getMonth() + 1;
                periodDisplay = new Date(currentDate.getFullYear(), displayMonth - 1)
                    .toLocaleString('default', { month: 'long' });
            }

            resolve({
                series: [{
                    name: 'Total Amount',
                    data: [
                        amounts.regular,
                        amounts.agent,
                        amounts.api
                    ]
                }],
                categories: ['Regular Users', 'Agent Users', 'API Users'],
                meta: {
                    period: periodDisplay,
                    month: month === 'all-time' ? null : (month || currentDate.getMonth() + 1),
                    year: currentDate.getFullYear()
                }
            });
        });
    });
};

//GET USER REGISTRATIONS BY YEAR
exports.getUserRegistrationsByYear = (year = null) => {
    return new Promise((resolve, reject) => {
        let query, params, meta;

        if (year === 'all-time') {
            // All-time query (monthly averages across all years)
            query = `
                SELECT 
                    MONTH(created_at) AS month,
                    FLOOR(COUNT(*) / COUNT(DISTINCT YEAR(created_at))) AS avg_users
                FROM f_users
                GROUP BY MONTH(created_at)
                ORDER BY month ASC
            `;
            meta = { period: 'All Time' };
        } else {
            // Year-specific query
            const targetYear = year || new Date().getFullYear();
            query = `
                SELECT 
                    MONTH(created_at) AS month,
                    COUNT(*) AS user_count
                FROM f_users
                WHERE YEAR(created_at) = ?
                GROUP BY MONTH(created_at)
                ORDER BY month ASC
            `;
            params = [targetYear];
            meta = { year: targetYear };
        }

        db.query(query, params, (err, results) => {
            if (err) return reject(err);
            
            const monthlyData = Array(12).fill(0);
            
            results.forEach(row => {
                monthlyData[row.month - 1] = year === 'all-time' ? row.avg_users : row.user_count;
            });
            
            resolve({
                series: [{
                    name: year === 'all-time' ? 'Avg. Users' : 'Users',
                    data: monthlyData
                }],
                categories: ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 
                           'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'],
                meta: meta
            });
        });
    });
};

//GET TRAFFICS
exports.getWebsiteTraffics = (date = null) => {
    return new Promise((resolve, reject) => {

        let query, params = [];
        
        if (date) {
            query = `SELECT hour, total_requests as total, successful_requests as success, failed_requests as errors FROM f_website_traffics WHERE date = ? ORDER BY hour ASC`;
            params = [date];
        } else {
            query = `SELECT hour, total_requests as total, successful_requests as success, failed_requests as errors FROM f_website_traffics  WHERE date = CURDATE() ORDER BY hour ASC`;
        } 

        db.query(query, params, (err, results) => {
            if (err) return reject(err);
            
            // Fill empty hours with zeros
            const hours = Array.from({ length: 24 }, (_, i) => i);
            const data = hours.map((h) => {
              const record = results.find((r) => r.hour === h) || {
                total: 0,
                success: 0,
                errors: 0,
              };
              return { hour: h, ...record };
            });
            
            resolve(data);
        });
    });
};

//GET MONTHLY REVENUE
exports.getMonthlyRevenue = () => {
    return new Promise((resolve, reject) => {

        db.query(`SELECT SUM(CASE WHEN st_type = 'deposit' THEN st_final_amount - st_amount WHEN st_type = 'withdrawal' THEN st_amount - st_final_amount ELSE st_final_amount - st_amount END) AS total FROM f_services_transactions WHERE st_status = 'successful' AND st_created_at >= DATE_FORMAT(CURRENT_DATE(), '%Y-%m-01') AND st_created_at < DATE_FORMAT(DATE_ADD(CURRENT_DATE(), INTERVAL 1 MONTH), '%Y-%m-01')`, (err, data) => {
            if (err) return reject(err);
            
            resolve(data[0].total);
        });
    });
};