const db = require("../models/db");
const { containsBannedWord } = require("./dataManipulator");

//CREATE NEW SCRAPPED CONTENT
exports.createScrappedContent = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_scrapped_contents SET ?", obj, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//CREATE NEW POST CATEGORY
exports.createPostCategory = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_content_categories SET ?", obj, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

// CREATE NEW POST CATEGORY PIVOT - BULK INSERT
exports.createPostCategoryPivot = (contentId, categoryIds) => {
    return new Promise((resolve, reject) => {
        if (!categoryIds.length) return resolve(); // nothing to insert

        const values = categoryIds.map(catId => [contentId, catId]);
        const sql = "INSERT INTO f_content_category_pivot (ccp_content_id, ccp_category_id) VALUES ?";

        db.query(sql, [values], (err, data) => {
            if (err) reject(err);
            else resolve(data);
        });
    });
};

// DELETE MULTIPLE CATEGORIES
exports.deletePostCategoryPivot = (categoryIds) => {
    return new Promise((resolve, reject) => {
        // Check if categoryIds is an array and not empty
        if (!Array.isArray(categoryIds) || categoryIds.length === 0) {
            return resolve(0); // Return 0 if no users to delete
        }

        // Convert all IDs to numbers to prevent SQL injection
        const sanitizedIds = categoryIds.map(id => parseInt(id)).filter(id => !isNaN(id));
        
        if (sanitizedIds.length === 0) {
            return resolve(0); // Return 0 if no valid IDs
        }

        // Create placeholders for the query (?, ?, ? etc.)
        const placeholders = sanitizedIds.map(() => '?').join(',');
        
        db.query(
            `DELETE FROM f_content_category_pivot WHERE ccp_category_id IN (${placeholders})`,
            sanitizedIds,
            (err, result) => {
                if (err) {
                    reject(err);
                } else {
                    // Return the number of affected (deleted) rows
                    resolve(result.affectedRows);
                }
            }
        );
    });
};

//EDIT POST CATEGORY
exports.editPostCategory = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("UPDATE f_content_categories SET ? WHERE cc_id = ?", [obj.data, obj.categoryId], (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

// DELETE MULTIPLE CATEGORIES
exports.deletePostCategories = (categoriesId) => {
    return new Promise((resolve, reject) => {
        // Check if categoriesId is an array and not empty
        if (!Array.isArray(categoriesId) || categoriesId.length === 0) {
            return resolve(0); // Return 0 if no users to delete
        }

        // Convert all IDs to numbers to prevent SQL injection
        const sanitizedIds = categoriesId.map(id => parseInt(id)).filter(id => !isNaN(id));
        
        if (sanitizedIds.length === 0) {
            return resolve(0); // Return 0 if no valid IDs
        }

        // Create placeholders for the query (?, ?, ? etc.)
        const placeholders = sanitizedIds.map(() => '?').join(',');
        
        db.query(
            `SELECT * FROM f_content_categories WHERE cc_id IN (${placeholders})`,
            sanitizedIds,
            (selectErr, rowsToDelete) => {
                if (selectErr) {
                    return reject(selectErr);
                }

                if (rowsToDelete.length === 0) {
                    return resolve({ deletedRows: 0, deletedData: [] });
                }

                db.query(
            `DELETE FROM f_content_categories WHERE cc_id IN (${placeholders})`,
            sanitizedIds,
            (err, result) => {
                if (err) {
                    reject(err);
                } else {
                    // Return the number of affected (deleted) rows
                    resolve({ deletedRows: result.affectedRows, deletedData: rowsToDelete });
                }
            }
        );
        })
        
    });
};

//GET POST CATEGORIES
exports.getPostCategories = (obj) => {
    return new Promise(async (resolve, reject) => {
        try {
            // Build dynamic filters
            let filters = `WHERE 1 + 1`;

            if (obj.query.search) {
                filters += ` AND (cc_name LIKE '%${obj.query.search}%' OR cc_slug LIKE '%${obj.query.search}%')`;
            }

            // Build SQL queries for pagination
            const query = `SELECT c.*, COUNT(p.ccp_id) AS postCount FROM f_content_categories c LEFT JOIN f_content_category_pivot p ON p.ccp_category_id = c.cc_id ${filters} GROUP BY c.cc_id ORDER BY c.cc_id DESC LIMIT ${obj.query.limit} OFFSET ${obj.query.offset}`;
            
            const countQuery = `SELECT COUNT(*) as totalCount FROM f_content_categories ${filters};`;

            const dbInstance = obj.connection || db;

            const [dataResult, countResult] = await Promise.all([
                new Promise((res, rej) => dbInstance.query(query, (err, data) => err ? rej(err) : res(data))),
                new Promise((res, rej) => dbInstance.query(countQuery, (err, data) => err ? rej(err) : res(data))),
            ]);

            return resolve({
                categories: dataResult,
                totalCount: countResult[0].totalCount
            });

        } catch (err) {
            return reject(err);
        }
    });
};

//GET POST CATEGORY BY ID
exports.getPostCategoryById = (catId) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_content_categories WHERE cc_id = ?", catId, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
}
//CREATE NEW POST
exports.createNewPost = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("INSERT INTO f_contents SET ?", obj, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//EDIT POST BY ID
exports.editPostById = (obj) => {
    return new Promise((resolve, reject) => {
        db.query("UPDATE f_contents SET ? WHERE c_id = ?", [obj.data, obj.postId], (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//UPDATE POST VIEW
exports.updatePostViews = (slug) => {
    return new Promise((resolve, reject) => {
        db.query("UPDATE f_contents SET c_views = c_views + 1 WHERE c_slug = ?", slug, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

//UPDATE POST REACTION
exports.updatePostReaction = (slug, type, action) => {
  let query = "";
  if(action){
    if(action == "like"){
      query = "UPDATE f_contents SET c_likes = c_likes - 1, c_dislikes = c_dislikes + 1 WHERE c_slug = ?";
    }else if(action == "dislike"){
      query = "UPDATE f_contents SET c_likes = c_likes + 1, c_dislikes = c_dislikes - 1 WHERE c_slug = ?";
    }
  }else{
    if(type == "like"){
      query = "UPDATE f_contents SET c_likes = c_likes + 1 WHERE c_slug = ?";
    }else if(type == "dislike"){
      query = "UPDATE f_contents SET c_dislikes = c_dislikes + 1 WHERE c_slug = ?";
    }
  }
  
  return new Promise((resolve, reject) => {
    db.query(query, slug, (err, data) => {
        if (err) reject(err)
        else resolve(data)
    })
  })
};

//GET POST
exports.getAllPost = (obj) => {
    return new Promise(async (resolve, reject) => {
        try {
            // Build dynamic filters
            let filters = `WHERE 1 = 1`;

            if (obj.query.search) {
                filters += ` AND (c.c_title LIKE '%${obj.query.search}%' OR c.c_slug LIKE '%${obj.query.search}%')`;
            };

            if (obj.query.categories && Array.isArray(obj.query.categories) && obj.query.categories.length > 0) {
                const catIds = obj.query.categories.map(id => db.escape(id)).join(',');
                filters += ` AND EXISTS (
                    SELECT 1 FROM f_content_category_pivot cp
                    WHERE cp.ccp_content_id = c.c_id AND cp.ccp_category_id IN (${catIds})
                )`;
            }

            if (obj.query.status) {
                const validStatuses = ['published', 'draft'];
                if (validStatuses.includes(obj.query.status)) {
                    filters += ` AND c.c_status = '${obj.query.status}'`;
                }
            }

            const query = `SELECT c.*, (SELECT CONCAT('[', GROUP_CONCAT(CONCAT('{"cc_id":', cat.cc_id, ',"cc_name":"', cat.cc_name, '","cc_slug":"', cat.cc_slug, '"}')), ']') FROM f_content_category_pivot pivot JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id WHERE pivot.ccp_content_id = c.c_id) AS categories FROM f_contents c ${filters} ORDER BY c.c_id DESC LIMIT ${obj.query.limit} OFFSET ${obj.query.offset}`;

            const countQuery = `SELECT COUNT(*) as totalCount FROM f_contents c ${filters}`;

            const dbInstance = obj.connection || db;

            const [dataResult, countResult] = await Promise.all([
                new Promise((res, rej) => dbInstance.query(query, (err, data) => err ? rej(err) : res(data))),
                new Promise((res, rej) => dbInstance.query(countQuery, (err, data) => err ? rej(err) : res(data))),
            ]);

            return resolve({
                contents: dataResult,
                totalCount: countResult[0].totalCount
            });

        } catch (err) {
            return reject(err);
        }
    });
};

//GET POST BY ID
exports.getPostById = (postId) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_contents WHERE c_id = ?", postId, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};

//GET YOU MAY ALSO LIKE POSTS
exports.getYouMayAlsoLikePosts = (postSlug, limit = 5) => {
  return new Promise((resolve, reject) => {
    // Step 1: Get the post details and its categories
    db.query(
      `SELECT 
        c.c_id, 
        c.c_title, 
        c.c_description,
        (
          SELECT GROUP_CONCAT(ccp_category_id)
          FROM f_content_category_pivot 
          WHERE ccp_content_id = c.c_id
        ) AS category_ids
      FROM f_contents c 
      WHERE c.c_slug = ?`,
      [postSlug],
      (err, result) => {
        if (err) return reject(err);
        if (!result.length) return resolve([]); // post not found

        const { c_id: postId, c_title: postTitle, c_description: postDescription, category_ids } = result[0];
        const postCategories = category_ids ? category_ids.split(',').map(Number) : [];
        
        // Step 2: Extract and clean keywords from title and description
        const allText = `${postTitle} ${postDescription}`;
        const keywords = allText
          .split(/\s+/)
          .map(w => w.trim().toLowerCase())
          .filter(w => w.length > 3) // remove short words
          .filter(w => !['and', 'the', 'how', 'what', 'when', 'where', 'why', 'for', 'with', 'this', 'that'].includes(w)) // remove common words
          .filter((w, i, a) => a.indexOf(w) === i) // remove duplicates
          .map(keyword => keyword.replace(/['"\\]/g, '')); // remove problematic characters

        // Function to safely get posts with categories
        const getPostsWithCategories = (query, params) => {
          return new Promise((resolve, reject) => {
            db.query(
              `SELECT 
                c.*,
                (
                  SELECT CONCAT('[',  
                    GROUP_CONCAT(
                      CONCAT(
                        '{"cc_id":', cat.cc_id,
                        ',"cc_name":"', REPLACE(cat.cc_name, '"', '\\"'),
                        '","cc_slug":"', REPLACE(cat.cc_slug, '"', '\\"'), '"}'
                      )
                    ), ']'
                  )
                  FROM f_content_category_pivot pivot
                  JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id
                  WHERE pivot.ccp_content_id = c.c_id AND c.c_status = 'published'
                ) AS categories
              ${query}`,
              params,
              (err, data) => err ? reject(err) : resolve(data)
            );
          });
        };

        // Safe function to find related posts by keywords using parameterized queries
        const findRelatedByKeywords = () => {
          if (!keywords.length) return Promise.resolve([]);

          // Create parameterized LIKE conditions
          const likeConditions = keywords
            .map(() => `(c.c_title LIKE ? OR c.c_description LIKE ?)`)
            .join(' OR ');

          // Create parameters for each keyword (each appears twice - for title and description)
          const likeParams = keywords.flatMap(keyword => [
            `%${keyword}%`,
            `%${keyword}%`
          ]);

          return getPostsWithCategories(
            `FROM f_contents c
            WHERE 
              c.c_status = 'published' AND 
              c.c_id != ? AND
              (${likeConditions})
            ORDER BY (
              ${keywords.map(() => `(CASE WHEN c.c_title LIKE ? THEN 2 ELSE 0 END) + (CASE WHEN c.c_description LIKE ? THEN 1 ELSE 0 END)`).join(' + ')}
            ) DESC, c.c_created_at DESC
            LIMIT ?`,
            [postId, ...likeParams, ...likeParams, limit] // params for both WHERE and ORDER BY
          );
        };

        // Find related posts by categories
        const findRelatedByCategories = (excludeIds = []) => {
          if (!postCategories.length) return Promise.resolve([]);

          const excludeClause = excludeIds.length ? `AND c.c_id NOT IN (${excludeIds.join(',')})` : '';
          
          return getPostsWithCategories(
            `FROM f_contents c
            JOIN f_content_category_pivot cp ON c.c_id = cp.ccp_content_id
            WHERE 
              c.c_id != ? AND c.c_status = 'published' AND 
              cp.ccp_category_id IN (${postCategories.join(',')})
              ${excludeClause}
            GROUP BY c.c_id
            ORDER BY COUNT(cp.ccp_category_id) DESC, c.c_created_at DESC
            LIMIT ?`,
            [postId, limit]
          );
        };

        // Final fallback to random posts
        const findRandomPosts = (excludeIds = []) => {
          const excludeClause = excludeIds.length ? `AND c.c_id NOT IN (${excludeIds.join(',')})` : '';
          
          return getPostsWithCategories(
            `FROM f_contents c
            WHERE 
              c.c_status = 'published' AND
              c.c_id != ?
              ${excludeClause}
            ORDER BY RAND()
            LIMIT ?`,
            [postId, limit]
          );
        };

        // Execute the queries in sequence
        findRelatedByKeywords()
          .then(relatedPosts => {
            const relatedCount = relatedPosts.length;
            const remainingCount = limit - relatedCount;
            const excludeIds = [...relatedPosts.map(p => p.c_id), postId];

            if (remainingCount <= 0) return resolve(relatedPosts);

            // Try to fill with category-matched posts first
            findRelatedByCategories(excludeIds)
              .then(categoryPosts => {
                const combined = [...relatedPosts, ...categoryPosts];
                const newExcludeIds = [...excludeIds, ...categoryPosts.map(p => p.c_id)];
                const newRemainingCount = limit - combined.length;

                if (newRemainingCount <= 0) return resolve(combined);

                // Fill any remaining slots with random posts
                findRandomPosts(newExcludeIds)
                  .then(randomPosts => {
                    resolve([...combined, ...randomPosts.slice(0, newRemainingCount)]);
                  })
                  .catch(() => resolve(combined)); // If error, return what we have
              })
              .catch(() => {
                // If category query fails, try random fallback
                findRandomPosts(excludeIds)
                  .then(randomPosts => {
                    resolve([...relatedPosts, ...randomPosts.slice(0, remainingCount)]);
                  })
                  .catch(() => resolve(relatedPosts)); // If error, return what we have
              });
          })
          .catch(err => reject(err));
      }
    );
  });
};



//GET POST BY ID
exports.getPostDetailsBySlug = (postSlug, fromAdmin = false) => {
    return new Promise((resolve, reject) => {
        db.query(`SELECT c.*, a.*, ( SELECT CONCAT('[', GROUP_CONCAT( CONCAT( '{"cc_id":', cat.cc_id, ',"cc_name":"', REPLACE(cat.cc_name, '"', '\\"'), '","cc_slug":"', REPLACE(cat.cc_slug, '"', '\\"'), '"}' ) ), ']' ) FROM f_content_category_pivot pivot JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id WHERE pivot.ccp_content_id = c.c_id ) AS categories FROM f_contents c LEFT JOIN f_admins a ON a.uid = c.c_author WHERE c_slug = ? AND ${fromAdmin ? 1 + 1 : `c_status = 'published'`}`, postSlug, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};

//GET CATEGORY BY SLUG
exports.getCategoryBySlug = (catSlug) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_content_categories WHERE cc_slug = ? AND cc_visibility = 'visible'", catSlug, (err, data) => {
            if (err) reject(err)
            else resolve(data[0])
        })
    })
};

// DELETE MULTIPLE POST
exports.deletePosts = (postIds) => {
    return new Promise((resolve, reject) => {
        // Validate input
        if (!Array.isArray(postIds) || postIds.length === 0) {
            return resolve({ deletedRows: 0, deletedData: [] });
        }

        const sanitizedIds = postIds.map(id => parseInt(id)).filter(id => !isNaN(id));

        if (sanitizedIds.length === 0) {
            return resolve({ deletedRows: 0, deletedData: [] });
        }

        const placeholders = sanitizedIds.map(() => '?').join(',');

        // Step 1: Select data before deletion
        db.query(
            `SELECT * FROM f_contents WHERE c_id IN (${placeholders})`,
            sanitizedIds,
            (selectErr, rowsToDelete) => {
                if (selectErr) {
                    return reject(selectErr);
                }

                if (rowsToDelete.length === 0) {
                    return resolve({ deletedRows: 0, deletedData: [] });
                }

                // Step 2: Delete the rows
                db.query(
                    `DELETE FROM f_contents WHERE c_id IN (${placeholders})`,
                    sanitizedIds,
                    (deleteErr, result) => {
                        if (deleteErr) {
                            return reject(deleteErr);
                        }

                        resolve({
                            deletedRows: result.affectedRows,
                            deletedData: rowsToDelete
                        });
                    }
                );
            }
        );
    });
};


//GET ALL CATEGORIES OF A POST
exports.getCategoriesOfPost = (postId) => {
    return new Promise((resolve, reject) => {
        db.query("SELECT * FROM f_content_category_pivot JOIN f_content_categories ON cc_id = ccp_category_id WHERE ccp_content_id = ?", postId, (err, data) => {
            if (err) reject(err)
            else resolve(data)
        })
    })
};

exports.getSmartLatestPost = (obj) => {
  return new Promise((resolve, reject) => {
    const limit = parseInt(obj.query.limit || 20);
    const offset = parseInt(obj.query.offset || 0);

    const dbInstance = db;

    const categoriesSubquery = `
      (SELECT CONCAT('[', 
        GROUP_CONCAT(
          CONCAT(
            '{"cc_id":', cat.cc_id,
            ',"cc_name":"', REPLACE(cat.cc_name, '"', '\\"'),
            '","cc_slug":"', REPLACE(cat.cc_slug, '"', '\\"'), '"}'
          )
        ), ']')
      FROM f_content_category_pivot pivot
      JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id
      WHERE pivot.ccp_content_id = c_id) AS categories
    `;

    // Get total count
    const countQuery = `
      SELECT COUNT(*) AS total
      FROM f_contents
      WHERE c_status = 'published'
    `;

    dbInstance.query(countQuery, (err, countResult) => {
      if (err) return reject(err);
      const totalCount = countResult[0].total;

      // Get paginated posts
      const query = `
        SELECT *, ${categoriesSubquery}
        FROM f_contents
        WHERE c_status = 'published'
        ORDER BY c_created_at DESC
        LIMIT ? OFFSET ?
      `;

      dbInstance.query(query, [limit, offset], (err, posts) => {
        if (err) return reject(err);

        resolve({
          posts,
          totalCount,
        });
      });
    });
  });
};




exports.getSmartLatestPostSearch = (obj) => {
  return new Promise(async (resolve, reject) => {
    try {
      const dbInstance = obj.connection || db;
      const { search, production, quality, duration, sortBy, limit, offset } = obj.query;

      // Base filter for published posts
      let filters = `WHERE c_status = 'published'`;

      const escapedSearch = `%${search}%`;
      filters += ` AND (c_title LIKE ${dbInstance.escape(escapedSearch)} OR c_description LIKE ${dbInstance.escape(escapedSearch)})`;

      if (production && production !== 'any') {
        filters += ` AND c_production = ${dbInstance.escape(production)}`;
      }
      
      if (quality && quality !== 'any') {
        filters += ` AND c_quality = ${dbInstance.escape(quality)}`;
      }

      if (duration && duration !== 'any') {
        if (duration === 'short') {
          filters += ` AND c_duration < 300`;
        } else if (duration === 'medium') {
          filters += ` AND c_duration >= 300 AND c_duration <= 1200`;
        } else if (duration === 'long') {
          filters += ` AND c_duration > 1200`;
        }
      }

      // Categories subquery
      const categoriesSubquery = `
        (SELECT CONCAT('[',
          GROUP_CONCAT(
            CONCAT(
              '{"cc_id":', cat.cc_id,
              ',"cc_name":"', REPLACE(cat.cc_name, '"', '\\"'),
              '","cc_slug":"', REPLACE(cat.cc_slug, '"', '\\"'), '"}'
            )
          ), ']')
        FROM f_content_category_pivot pivot
        JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id
        WHERE pivot.ccp_content_id = c_id) AS categories
      `;

      // Sorting logic
      let orderBy = `ORDER BY c_created_at DESC`;
      if (sortBy && sortBy !== 'any') {
        switch (sortBy) {
          case 'trending':
            orderBy = `ORDER BY (c_likes * 2 + c_views) DESC`;
            break;
          case 'latest':
            orderBy = `ORDER BY c_created_at DESC`;
            break;
          case 'viewed':
            orderBy = `ORDER BY c_views DESC`;
            break;
          case 'liked':
            orderBy = `ORDER BY c_likes DESC`;
            break;
        }
      }

      // Get total count
      const countQuery = `SELECT COUNT(*) AS totalCount FROM f_contents ${filters}`;
      const countResult = await new Promise((res, rej) => {
        dbInstance.query(countQuery, (err, data) => err ? rej(err) : res(data));
      });

      // Get paginated data
      const dataQuery = `SELECT *, ${categoriesSubquery} FROM f_contents ${filters} ${orderBy} LIMIT ${parseInt(limit)} OFFSET ${parseInt(offset)}`;
      const posts = await new Promise((res, rej) => {
        dbInstance.query(dataQuery, (err, data) => err ? rej(err) : res(data));
      });

      resolve({
        posts,
        totalCount: countResult[0]?.totalCount || 0
      });

    } catch (err) {
      reject(err);
    }
  });
};

exports.getPostsByTag = (obj) => {
  return new Promise(async (resolve, reject) => {
    try {
      const dbInstance = obj.connection || db;
      const { tag, production, quality, duration, sortBy, limit, offset } = obj.query;

      // Base filter for published posts
      let filters = `WHERE c_status = 'published'`;

      // Filter by tag using JSON_CONTAINS
      if (tag) {
        filters += ` AND JSON_CONTAINS(c_tags, ${dbInstance.escape(`"${tag}"`)})`;
      }

      // Optional filters
      if (production && production !== 'any') {
        filters += ` AND c_production = ${dbInstance.escape(production)}`;
      }

      if (quality && quality !== 'any') {
        filters += ` AND c_quality = ${dbInstance.escape(quality)}`;
      }

      if (duration && duration !== 'any') {
        if (duration === 'short') {
          filters += ` AND c_duration < 300`;
        } else if (duration === 'medium') {
          filters += ` AND c_duration >= 300 AND c_duration <= 1200`;
        } else if (duration === 'long') {
          filters += ` AND c_duration > 1200`;
        }
      }

      // Categories subquery for each post
      const categoriesSubquery = `
        (SELECT CONCAT('[',
          GROUP_CONCAT(
            CONCAT(
              '{"cc_id":', cat.cc_id,
              ',"cc_name":"', REPLACE(cat.cc_name, '"', '\\"'),
              '","cc_slug":"', REPLACE(cat.cc_slug, '"', '\\"'), '"}'
            )
          ), ']')
        FROM f_content_category_pivot pivot
        JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id
        WHERE pivot.ccp_content_id = c_id) AS categories
      `;

      // Sorting logic
      let orderBy = `ORDER BY c_created_at DESC`;
      if (sortBy && sortBy !== 'any') {
        switch (sortBy) {
          case 'trending':
            orderBy = `ORDER BY (c_likes * 2 + c_views) DESC`;
            break;
          case 'latest':
            orderBy = `ORDER BY c_created_at DESC`;
            break;
          case 'viewed':
            orderBy = `ORDER BY c_views DESC`;
            break;
          case 'liked':
            orderBy = `ORDER BY c_likes DESC`;
            break;
        }
      }

      // Get total count
      const countQuery = `SELECT COUNT(*) AS totalCount FROM f_contents ${filters}`;
      const countResult = await new Promise((res, rej) => {
        dbInstance.query(countQuery, (err, data) => err ? rej(err) : res(data));
      });

      // Get paginated data
      const dataQuery = `SELECT *, ${categoriesSubquery} FROM f_contents ${filters} ${orderBy} LIMIT ${parseInt(limit)} OFFSET ${parseInt(offset)}`;
      const posts = await new Promise((res, rej) => {
        dbInstance.query(dataQuery, (err, data) => err ? rej(err) : res(data));
      });

      resolve({
        posts,
        totalCount: countResult[0]?.totalCount || 0
      });

    } catch (err) {
      reject(err);
    }
  });
};





exports.getHotTrendingPosts = (obj) => {
    return new Promise((resolve, reject) => {
      const dbInstance = db;
  
      const limit = obj.query.limit || 20;
      const offset = obj.query.offset || 0;
  
      const whereClause = `WHERE c_status = 'published'`;
  
      const categoriesSubquery = `
        (
          SELECT CONCAT('[', 
            GROUP_CONCAT(
              CONCAT(
                '{"cc_id":', cat.cc_id,
                ',"cc_name":"', REPLACE(cat.cc_name, '"', '\\"'),
                '","cc_slug":"', REPLACE(cat.cc_slug, '"', '\\"'), '"}'
              )
            ), ']')
          FROM f_content_category_pivot pivot
          JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id
          WHERE pivot.ccp_content_id = f.c_id
        ) AS categories
      `;
  
      const dataQuery = `
        SELECT 
          f.*,
          ${categoriesSubquery},
          (c_views + c_likes * 2) / POW(TIMESTAMPDIFF(HOUR, c_created_at, NOW()) + 2, 1.5) AS hot_score
        FROM f_contents f
        ${whereClause}
        ORDER BY hot_score DESC
        LIMIT ? OFFSET ?;
      `;
  
      const countQuery = `SELECT COUNT(*) AS totalCount FROM f_contents ${whereClause};`;
  
      Promise.all([
        new Promise((res, rej) => {
          dbInstance.query(dataQuery, [limit, offset], (err, posts) => {
            if (err) return rej(err);
            res(posts);
          });
        }),
        new Promise((res, rej) => {
          dbInstance.query(countQuery, (err, countRows) => {
            if (err) return rej(err);
            res(countRows[0]?.totalCount || 0);
          });
        })
      ])
      .then(([posts, totalCount]) => {
        resolve({
          posts: posts || [],
          totalCount
        });
      })
      .catch(reject);
    });
};

exports.getHotTrendingPosts2 = (obj) => {
  return new Promise((resolve, reject) => {
    const dbInstance = db;

    const limit = obj.query.limit || 20;
    const offset = obj.query.offset || 0;

    // WHERE clause includes filtering for the current month
    const whereClause = `
      WHERE c_status = 'published'
      AND MONTH(c_created_at) = MONTH(CURRENT_DATE())
      AND YEAR(c_created_at) = YEAR(CURRENT_DATE())
    `;

    const categoriesSubquery = `
      (
        SELECT CONCAT('[', 
          GROUP_CONCAT(
            CONCAT(
              '{"cc_id":', cat.cc_id,
              ',"cc_name":"', REPLACE(cat.cc_name, '"', '\\"'),
              '","cc_slug":"', REPLACE(cat.cc_slug, '"', '\\"'), '"}'
            )
          ), ']')
        FROM f_content_category_pivot pivot
        JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id
        WHERE pivot.ccp_content_id = f.c_id
      ) AS categories
    `;

    // Hot score formula — still based on engagement vs. freshness
    const hotScoreFormula = `
      (c_views + c_likes * 2) / POW(TIMESTAMPDIFF(HOUR, c_created_at, NOW()) + 2, 1.5)
    `;

    const dataQuery = `
      SELECT 
        f.*,
        ${categoriesSubquery},
        ${hotScoreFormula} AS hot_score
      FROM f_contents f
      ${whereClause}
      ORDER BY hot_score DESC
      LIMIT ? OFFSET ?;
    `;

    const countQuery = `
      SELECT COUNT(*) AS totalCount 
      FROM f_contents 
      ${whereClause};
    `;

    Promise.all([
      new Promise((res, rej) => {
        dbInstance.query(dataQuery, [limit, offset], (err, posts) => {
          if (err) return rej(err);
          res(posts);
        });
      }),
      new Promise((res, rej) => {
        dbInstance.query(countQuery, (err, countRows) => {
          if (err) return rej(err);
          res(countRows[0]?.totalCount || 0);
        });
      })
    ])
    .then(([posts, totalCount]) => {
      resolve({
        posts: posts || [],
        totalCount
      });
    })
    .catch(reject);
  });
};


exports.getTrendingPostsLast24Hours = (obj) => {
    return new Promise((resolve, reject) => {
      const dbInstance = db;
  
      const limit = obj.query.limit || 20;
      const offset = obj.query.offset || 0;
  
      const categoriesSubquery = `
        (
          SELECT CONCAT('[', 
            GROUP_CONCAT(
              CONCAT(
                '{"cc_id":', cat.cc_id,
                ',"cc_name":"', REPLACE(cat.cc_name, '"', '\\"'),
                '","cc_slug":"', REPLACE(cat.cc_slug, '"', '\\"'), '"}'
              )
            ), ']')
          FROM f_content_category_pivot pivot
          JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id
          WHERE pivot.ccp_content_id = f.c_id
        ) AS categories
      `;
  
      const dataQuery = `
        SELECT 
          f.*,
          ${categoriesSubquery}
        FROM f_contents f
        WHERE c_status = 'published'
          AND c_created_at >= NOW() - INTERVAL 1 DAY
          AND (c_views > 0 OR c_likes > 0)
        ORDER BY GREATEST(c_views, c_likes) DESC
        LIMIT ? OFFSET ?;
      `;
  
      const countQuery = `
        SELECT COUNT(*) AS totalCount
        FROM f_contents
        WHERE c_status = 'published'
          AND c_created_at >= NOW() - INTERVAL 1 DAY
          AND (c_views > 0 OR c_likes > 0);
      `;
  
      Promise.all([
        new Promise((res, rej) => {
          dbInstance.query(dataQuery, [limit, offset], (err, posts) => {
            if (err) return rej(err);
            res(posts);
          });
        }),
        new Promise((res, rej) => {
          dbInstance.query(countQuery, (err, data) => {
            if (err) return rej(err);
            res(data[0]?.totalCount || 0);
          });
        })
      ])
      .then(([posts, totalCount]) => {
        resolve({
          posts: posts || [],
          totalCount
        });
      })
      .catch(reject);
    });
  };
  
  
/*  */
exports.getTrendingCategoryPosts = async (fallbackCats = [], totalCatsLimit, postLimit = 20) => {
    try {
      // 1. First get trending categories
      const trendingCategories = await this.getTrendingCategories(totalCatsLimit);
      
      // 2. If no trending categories, use fallback
      const categoriesToProcess = trendingCategories.length > 0 ? trendingCategories : await Promise.all(
      fallbackCats.map(async fc => {
        const category = await this.getPostCategoryById(fc.catId);
        return {
          ...category,
          cc_name: fc.title || category.cc_name,
        };
      })
    );

    // 3. Get posts for each category (max 10 posts per category)
    const categoriesWithPosts = await Promise.all(
        categoriesToProcess.map(async (category) => {
            const posts = await getPostsForCategory(category.cc_id, postLimit);
            return {
            ...category,
            posts
            };
        })
    );
  
    return categoriesWithPosts;
    
    } catch (error) {
      console.error('Error in getTrendingCategoryPosts:', error);
      // Return fallback if any error occurs
      return [];
    }
  };
  
  // Helper function to get trending categories
  exports.getTrendingCategories = async function (limit = 5) {
    return new Promise((resolve, reject) => {
      const query = `
        SELECT 
          cat.cc_id,
          cat.cc_name, 
          cat.cc_slug,
          cat.cc_flag,
          COUNT(pivot.ccp_content_id) AS content_count,
          SUM(c.c_views) AS total_views,
          (SUM(c.c_views) * 0.7 + COUNT(pivot.ccp_content_id) * 0.3) AS popularity_score
        FROM f_content_categories cat
        JOIN f_content_category_pivot pivot ON cat.cc_id = pivot.ccp_category_id
        JOIN f_contents c ON pivot.ccp_content_id = c.c_id
        WHERE c.c_status = 'published'
          AND c.c_created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
        GROUP BY cat.cc_id, cat.cc_name, cat.cc_slug, cat.cc_flag
        ORDER BY popularity_score DESC
        LIMIT ?
      `;

      db.query(query, limit, (err, results) => {
        if (err) return reject(err);
        resolve(results || []);
      });
    });
  }
  
  // Helper function to get posts for a category
  async function getPostsForCategory(categoryId, limit = 10) {
    return new Promise((resolve, reject) => {
      const query = `
        SELECT 
          c.*,
          (
            SELECT CONCAT('[', 
              GROUP_CONCAT(
                CONCAT(
                  '{"cc_id":', cat.cc_id,
                  ',"cc_name":"', REPLACE(cat.cc_name, '"', '\\"'),
                  '","cc_slug":"', REPLACE(cat.cc_slug, '"', '\\"'), '"}'
                )
              ), ']'
            )
            FROM f_content_category_pivot pivot
            JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id
            WHERE pivot.ccp_content_id = c.c_id
          ) AS categories
        FROM f_contents c
        JOIN f_content_category_pivot pivot ON c.c_id = pivot.ccp_content_id
        WHERE pivot.ccp_category_id = ?
          AND c.c_status = 'published'
        ORDER BY c.c_created_at DESC
        LIMIT ?
      `;
  
      db.query(query, [categoryId, limit], (err, results) => {
        if (err) return reject(err);
        resolve(results || []);
      });
    });
  }

exports.getCategoryPosts = (obj) => {
  return new Promise(async (resolve, reject) => {
    try {
      const dbInstance = obj.connection || db;
      const { catId, search, production, quality, duration, sortBy, limit = 20, offset = 0 } = obj.query;

      if (!catId) {
        return reject(new Error('Category ID is required'));
      }

      const escapedCatId = dbInstance.escape(catId);
      const escapedSearch = `%${search}%`;

      // Filter: only posts that belong to the category (regardless of other categories)
      let filters = `WHERE fc.c_status = 'published' AND fc.c_id IN (SELECT ccp_content_id FROM f_content_category_pivot WHERE ccp_category_id = ${escapedCatId})`;

      if (search) {
        filters += ` AND (fc.c_title LIKE ${dbInstance.escape(escapedSearch)} OR fc.c_description LIKE ${dbInstance.escape(escapedSearch)})`;
      }else{

        if (production && production !== 'any') {
          filters += ` AND fc.c_production = ${dbInstance.escape(production)}`;
        };
        
        if (quality && quality !== 'any') {
          filters += ` AND fc.c_quality = ${dbInstance.escape(quality)}`;
        };

        if (duration && duration !== 'any') {
          if (duration == 'short') {
            filters += ` AND fc.c_duration < 300`;
          } else if (duration == 'medium') {
            filters += ` AND fc.c_duration >= 300 AND fc.c_duration <= 1200`;
          } else if (duration == 'long') {
            filters += ` AND fc.c_duration > 1200`;
          }
        }


      }

      

      // Categories JSON subquery (marking which is the current category)
      const categoriesSubquery = `
        (
          SELECT CONCAT(
            '[', 
            GROUP_CONCAT(
              CONCAT(
                '{"cc_id":', cat.cc_id,
                ',"cc_name":"', REPLACE(cat.cc_name, '"', '\\"'),
                '","cc_slug":"', REPLACE(cat.cc_slug, '"', '\\"'),
                '","is_current":', IF(cat.cc_id = ${escapedCatId}, 'true', 'false'), '}'
              )
              ORDER BY (cat.cc_id = ${escapedCatId}) DESC, cat.cc_id ASC
            ),
            ']'
          )
          FROM f_content_category_pivot pivot
          JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id
          WHERE pivot.ccp_content_id = fc.c_id
        ) AS categories
      `;

      // Default sort
      let orderBy = `ORDER BY fc.c_created_at DESC`;

      if (sortBy && sortBy !== 'any') {
        switch (sortBy) {
          case 'trending':
            // Define trending logic – example uses likes/views combined (adjust as needed)
            orderBy = `ORDER BY (fc.c_likes * 2 + fc.c_views) DESC`;
            break;
          case 'latest':
            orderBy = `ORDER BY fc.c_created_at DESC`;
            break;
          case 'viewed':
            orderBy = `ORDER BY fc.c_views DESC`;
            break;
          case 'liked':
            orderBy = `ORDER BY fc.c_likes DESC`;
            break;
        }
      }

      const dataQuery = `SELECT fc.*, ${categoriesSubquery} FROM f_contents fc ${filters} ${orderBy} LIMIT ${parseInt(limit)} OFFSET ${parseInt(offset)}`;

      const countQuery = `SELECT COUNT(*) AS totalCount FROM f_contents fc ${filters}`;

      const [posts, countResult] = await Promise.all([
        new Promise((res, rej) => dbInstance.query(dataQuery, (err, data) => err ? rej(err) : res(data))),
        new Promise((res, rej) => dbInstance.query(countQuery, (err, data) => err ? rej(err) : res(data)))
      ]);

      return resolve({
        posts,
        totalCount: countResult[0]?.totalCount || 0
      });

    } catch (err) {
      return reject(err);
    }
  });
};




exports.getTopRatedPosts = (obj) => {
  return new Promise((resolve, reject) => {
    const dbInstance = db;

    const limit = obj.query.limit || 20;
    const offset = obj.query.offset || 0;

    const whereClause = `
      WHERE c_status = 'published'
      AND MONTH(c_created_at) = MONTH(CURRENT_DATE())
      AND YEAR(c_created_at) = YEAR(CURRENT_DATE())
    `;

    const categoriesSubquery = `
      (
        SELECT CONCAT('[', 
          GROUP_CONCAT(
            CONCAT(
              '{"cc_id":', cat.cc_id,
              ',"cc_name":"', REPLACE(cat.cc_name, '"', '\\"'),
              '","cc_slug":"', REPLACE(cat.cc_slug, '"', '\\"'), '"}'
            )
          ), ']')
        FROM f_content_category_pivot pivot
        JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id
        WHERE pivot.ccp_content_id = f.c_id
      ) AS categories
    `;

    const dataQuery = `
      SELECT 
        f.*,
        ${categoriesSubquery}
      FROM f_contents f
      ${whereClause}
      ORDER BY c_likes DESC
      LIMIT ? OFFSET ?;
    `;

    const countQuery = `
      SELECT COUNT(*) AS totalCount 
      FROM f_contents 
      ${whereClause};
    `;

    Promise.all([
      new Promise((res, rej) => {
        dbInstance.query(dataQuery, [limit, offset], (err, posts) => {
          if (err) return rej(err);
          res(posts);
        });
      }),
      new Promise((res, rej) => {
        dbInstance.query(countQuery, (err, countRows) => {
          if (err) return rej(err);
          res(countRows[0]?.totalCount || 0);
        });
      })
    ])
    .then(([posts, totalCount]) => {
      resolve({
        posts: posts || [],
        totalCount
      });
    })
    .catch(reject);
  });
};

//GET ALL CATEGORY FOR USER
exports.getPostCategoriesForUser = () => {
    return new Promise((resolve, reject) => {
        db.query("SELECT cat.*, COUNT(c.c_id) AS post_count FROM f_content_categories AS cat INNER JOIN f_content_category_pivot AS pivot ON cat.cc_id = pivot.ccp_category_id INNER JOIN f_contents AS c ON pivot.ccp_content_id = c.c_id WHERE cat.cc_visibility = 'visible' AND c.c_status = 'published' GROUP BY cat.cc_id ORDER BY cat.cc_name ASC", (err, data) => {
            if (err) reject(err)
            else {
              const categories = [];
              const countries = [];

              data.forEach(item => {
                  if (item.cc_flag) {
                      countries.push(item);
                  } else {
                      categories.push(item);
                  }
              });

              return resolve({ categories, countries });
            }
        })
    })
}


exports.getPopularTagsThisMonth = (limit) => {
    return new Promise((resolve, reject) => {
        if (!db) {
            const error = new Error('Database connection not established');
            console.error(error);
            return reject(error);
        }

        const query = `
            SELECT 
                extracted_tag AS tag,
                COUNT(*) AS tag_count
            FROM (
                SELECT 
                    SUBSTRING_INDEX(
                        SUBSTRING_INDEX(
                            REPLACE(
                                REPLACE(
                                    REPLACE(
                                        COALESCE(c_tags, '[]'),
                                        '[', ''
                                    ),
                                    ']', ''
                                ),
                                '"', ''
                            ),
                            ',',
                            numbers.n
                        ),
                        ',',
                        -1
                    ) AS extracted_tag
                FROM 
                    f_contents
                JOIN 
                    (SELECT 1 AS n UNION SELECT 2 UNION SELECT 3 UNION 
                     SELECT 4 UNION SELECT 5 UNION SELECT 6 UNION 
                     SELECT 7 UNION SELECT 8 UNION SELECT 9 UNION 
                     SELECT 10) numbers
                    ON numbers.n <= (
                        LENGTH(COALESCE(c_tags, '[]')) - 
                        LENGTH(REPLACE(COALESCE(c_tags, '[]'), ',', '')) + 1
                    )
                WHERE 
                    c_status = 'published'
                    AND c_created_at >= DATE_FORMAT(CURRENT_DATE(), '%Y-%m-01')
                    AND c_created_at < DATE_FORMAT(CURRENT_DATE() + INTERVAL 1 MONTH, '%Y-%m-01')
                    AND COALESCE(c_tags, '[]') != '[]'
            ) AS extracted_tags
            WHERE 
                extracted_tag != ''
            GROUP BY 
                tag
            ORDER BY 
                tag_count DESC
            LIMIT ?;
        `;

        db.query(query, limit, (err, results) => {
            if (err) {
                console.error('Database query error:', err);
                return reject(err);
            }

            return resolve(results);
        });
    });
};



exports.bulkUpdatePosts = (postIds, updateFields) => {
    return new Promise((resolve, reject) => {
        if (!Array.isArray(postIds) || postIds.length === 0 || !updateFields || typeof updateFields !== 'object') {
            return resolve(0);
        }

        // Sanitize and validate IDs
        const sanitizedIds = postIds.map(id => parseInt(id)).filter(id => !isNaN(id));
        if (sanitizedIds.length === 0) {
            return resolve(0);
        }

        // Generate SET clause from updateFields
        const fieldKeys = Object.keys(updateFields);
        const setClause = fieldKeys.map(key => `${key} = ?`).join(', ');
        const values = fieldKeys.map(key => updateFields[key]);

        // Append IDs for WHERE IN clause
        const placeholders = sanitizedIds.map(() => '?').join(',');

        const sql = `UPDATE f_contents SET ${setClause} WHERE c_id IN (${placeholders})`;

        db.query(sql, [...values, ...sanitizedIds], (err, result) => {
            if (err) {
                reject(err);
            } else {
                resolve(result.affectedRows);
            }
        });
    });
};


exports.getPublishedContentsWithCategories = (limit, offset) => {
  return new Promise((resolve, reject) => {
    const query = `SELECT *, (SELECT CONCAT('[', GROUP_CONCAT( CONCAT( '{"cc_id":', cat.cc_id, ',"cc_name":"', REPLACE(cat.cc_name, '"', '\"'), '","cc_slug":"', REPLACE(cat.cc_slug, '"', '\"'), '"}' ) ), ']') FROM f_content_category_pivot pivot JOIN f_content_categories cat ON pivot.ccp_category_id = cat.cc_id WHERE pivot.ccp_content_id = c_id) AS categories FROM f_contents WHERE c_status = 'published' ORDER BY c_id DESC LIMIT ? OFFSET ?`;

    db.query(query, [limit, offset], (err, results) => {
      if (err) reject(err);
      else resolve(results);
    });
  });
};


//FILTER POST DATA
exports.filterPostData = (datas = { title: '', description: '', tags: '' }, parsedBannedWords = []) => {

  if(datas.title){
    const result = containsBannedWord(datas.title, parsedBannedWords);
    if(result.length) return { status: false, message: `Title contain some banned words:  ${result.join(",")}` }
  }
  
  if(datas.description){
    const result = containsBannedWord(datas.description, parsedBannedWords);
    if(result.length) return { status: false, message: `Description contain some banned words:  ${result.join(",")}` }
  }
  
  if(datas.tags){
    const result = containsBannedWord(datas.tags, parsedBannedWords);
    if(result.length) return { status: false, message: `Tags contain some banned words:  ${result.join(",")}` }
  }

  return { status: true }
}
