const db = require("../models/db");

//GET CONTENT
exports.getContentSitemapPagesWithLastMod = async () => {

  const PAGE_SIZE = 50000;
  const limit = (await import('p-limit')).default(Number(process.env.PROMISE_LIMIT || 5));

  // Helper: promisify db.query
  const query = (sql) =>
    new Promise((resolve, reject) => {
      db.query(sql, (err, data) => {
        if (err) reject(err);
        else resolve(data);
      });
    });

  try {
    // Step 1: Get total rows
    const countRows = await query('SELECT COUNT(*) AS total FROM f_contents');
    const total = countRows[0].total;
    const totalPages = Math.ceil(total / PAGE_SIZE);

    // Step 2: Build parallel tasks for each page
    const tasks = Array.from({ length: totalPages }, (_, i) => {
      const page = i + 1;
      const offset = i * PAGE_SIZE;

      return limit(async () => {
        const res = await query(`
          SELECT MAX(c_updated_at) AS lastmod
          FROM (
            SELECT c_updated_at
            FROM f_contents
            ORDER BY c_id ASC
            LIMIT ${PAGE_SIZE} OFFSET ${offset}
          ) AS sub
        `);

        const lastmod = res[0].lastmod
          ? new Date(res[0].lastmod).toISOString().split('T')[0]
          : new Date().toISOString().split('T')[0];

        return { page, lastmod };
      });
    });

    // Step 3: Run all in parallel (within concurrency limit)
    const results = await Promise.all(tasks);
    return results;
  } catch (error) {
    throw error;
  }
};

exports.getCategorySitemapPagesWithLastMod = async () => {

  const limit = (await import('p-limit')).default(Number(process.env.PROMISE_LIMIT || 5));

  const query = (sql, values = []) =>
    new Promise((resolve, reject) => {
      db.query(sql, values, (err, results) => {
        if (err) reject(err);
        else resolve(results);
      });
    });

  try {
    // Step 1: Get all categories with at least one post
    const categories = await query(`
      SELECT DISTINCT c.cc_id, c.cc_name, c.cc_description, c.cc_visibility, cc_thumbnail, c.cc_slug FROM f_content_categories c INNER JOIN f_content_category_pivot p ON c.cc_id = p.ccp_category_id INNER JOIN f_contents f ON f.c_id = p.ccp_content_id WHERE c.cc_visibility = 'visible' AND f.c_status = 'published' 
    `);
    
    // Step 2: Create parallel tasks to get lastmod per category
    const tasks = categories.map((item) =>
      limit(async () => {
        const res = await query(`
          SELECT MAX(c.c_updated_at) AS lastmod
          FROM f_contents c
          INNER JOIN f_content_category_pivot p ON c.c_id = p.ccp_content_id
          WHERE p.ccp_category_id = ? AND c.c_status = 'published'
        `, [item.cc_id]);

        //Getting Category Thumbnail
        let thumbnail;
        if(item.cc_thumbnail) thumbnail = item.cc_thumbnail;
        else{
          const thumbResult = await query(`
            SELECT c_thumbnail 
            FROM f_contents c
            INNER JOIN f_content_category_pivot p ON c.c_id = p.ccp_content_id
            WHERE p.ccp_category_id = ? AND c_thumbnail IS NOT NULL AND c_thumbnail != ''
            ORDER BY RAND() LIMIT 1
          `, [item.cc_id]);

          thumbnail = thumbResult[0]?.c_thumbnail || null
        }
        

        const lastmod = res[0].lastmod ? new Date(res[0].lastmod).toISOString().split('T')[0] : new Date().toISOString().split('T')[0];

        return { ...item, lastmod, thumbnail };
      })
    );

    const results = await Promise.all(tasks);
    return results;
  } catch (err) {
    throw err;
  }
};

exports.getTagSitemapPagesWithLastMod = async () => {
  const limit = (await import('p-limit')).default(Number(process.env.PROMISE_LIMIT || 5));

  const query = (sql, values = []) =>
    new Promise((resolve, reject) => {
      db.query(sql, values, (err, results) => {
        if (err) reject(err);
        else resolve(results);
      });
    });

  try {
    // Step 1: Get all published content with tags
    const rows = await query(`
      SELECT c_tags, c_updated_at 
      FROM f_contents 
      WHERE c_status = 'published' AND JSON_LENGTH(c_tags) > 0
    `);

    const tagMap = new Map();

    // Step 2: Process tags
    for (const row of rows) {
      const tags = JSON.parse(row.c_tags || '[]');
      const lastmod = new Date(row.c_updated_at).toISOString().split('T')[0];

      tags.forEach(tag => {
        tag = tag.trim().toLowerCase();

        // Basic SEO filters
        if (
          tag.length > 2 &&
          /^[a-z0-9\- ]+$/i.test(tag)
        ) {
          if (!tagMap.has(tag)) {
            tagMap.set(tag, { count: 1, lastmod });
          } else {
            const entry = tagMap.get(tag);
            entry.count++;
            entry.lastmod = entry.lastmod > lastmod ? entry.lastmod : lastmod; // use latest lastmod
            tagMap.set(tag, entry);
          }
        }
      });
    }

    // Step 3: Convert to array, sort by count descending
    const results = Array.from(tagMap.entries())
      .filter(([tag, data]) => data.count >= 1) // more than once = SEO worthy
      .sort((a, b) => b[1].count - a[1].count)
      .slice(0, 50000) // max 50k
      .map(([tag, data]) => ({
        tag,
        lastmod: data.lastmod,
      }));

    return results;
  } catch (error) {
    throw error;
  }
};


exports.getAllSitemapLastmods = async () => {
  const query = (sql, values = []) =>
    new Promise((resolve, reject) => {
      db.query(sql, values, (err, result) => {
        if (err) reject(err);
        else resolve(result[0]?.lastmod || null);
      });
    });

  try {
    const [videosLastmod, categoriesLastmod, pagesLastmod, tagsLastmod] = await Promise.all([
      query(`SELECT MAX(c_updated_at) AS lastmod FROM f_contents WHERE c_status = 'published'`),

      query(`
        SELECT MAX(c.c_updated_at) AS lastmod
        FROM f_contents c
        INNER JOIN f_content_category_pivot p ON c.c_id = p.ccp_content_id
        WHERE c.c_status = 'published'
      `),

      query(`SELECT MAX(page_updated_at) AS lastmod FROM f_pages`),

      query(`
        SELECT MAX(c_updated_at) AS lastmod 
        FROM f_contents 
        WHERE c_status = 'published' AND JSON_LENGTH(c_tags) > 0
      `),
    ]);

    return {
      videos: videosLastmod,
      categories: categoriesLastmod,
      pages: pagesLastmod,
      tags: tagsLastmod,
    };
  } catch (error) {
    throw error;
  }
};
