const ffmpegPath = require('@ffmpeg-installer/ffmpeg').path;
const ffprobePath = require('@ffprobe-installer/ffprobe').path;
const ffmpeg = require('fluent-ffmpeg');
const path = require('path');
const fs = require('fs');
const tmp = require('tmp');

ffmpeg.setFfmpegPath(ffmpegPath);
ffmpeg.setFfprobePath(ffprobePath);

// Configure temp cleanup
tmp.setGracefulCleanup();

const createTeaserVideo = async (obj) => {

    const { inputPath, outputDir, slug } = obj;

  // Verify input file exists
  if (!fs.existsSync(inputPath)) {
    throw new Error(`Input file not found: ${inputPath}`);
  }

  return new Promise((resolve, reject) => {
    const outputName = `teaser.mp4`;
    const outputPath = path.join(outputDir, outputName);

    ffmpeg.ffprobe(inputPath, async (err, metadata) => {
      if (err) return reject(err);

      const duration = metadata.format.duration;
      if (duration < 15) return reject(new Error('Video must be at least 15 seconds for 9s teaser'));

      const tempDir = tmp.dirSync({ unsafeCleanup: true }).name;
      const segmentLength = 3;
      const playbackSpeed = 0.5; // 2x speed (0.5 = 2x faster)
      const offsets = [5, Math.floor(duration * 0.4), Math.max(0, duration - 5 - segmentLength)];

      try {
        // 1. First extract all clips
        const clipPaths = await Promise.all(offsets.map((start, i) => {
          return new Promise((resolve, reject) => {
            const clipPath = path.join(tempDir, `clip-${i}.mp4`);
            
            ffmpeg(inputPath)
              .setStartTime(start)
              .duration(segmentLength)
              .noAudio()
              .videoFilters(`setpts=${playbackSpeed}*PTS`) // 2x speed
              .size('240x250')
              .output(clipPath)
              .on('end', () => resolve(clipPath))
              .on('error', reject)
              .run();
          });
        }))

        // 2. Verify all clips were created
        for (const clipPath of clipPaths) {
          if (!fs.existsSync(clipPath)) {
            throw new Error(`Failed to create clip: ${clipPath}`);
          }
        }

        // 3. Concatenate clips
        await new Promise((resolve, reject) => {
          ffmpeg()
            .input(clipPaths[0])
            .input(clipPaths[1])
            .input(clipPaths[2])
            .inputOptions([
              '-filter_complex',
              `[0:v][1:v][2:v]concat=n=3:v=1:a=0`
            ])
            .output(outputPath)
            .on('end', () => {
              fs.rmSync(tempDir, { recursive: true });
              resolve(outputPath);
            })
            .on('error', (err) => {
              fs.rmSync(tempDir, { recursive: true });
              reject(err);
            })
            .run();
        });

        resolve(`/files/videos/${slug}/${outputName}`);
      } catch (error) {
        if (fs.existsSync(tempDir)) {
          fs.rmSync(tempDir, { recursive: true });
        }
        reject(error);
      }
    });
  });
};

module.exports = { createTeaserVideo }